package at.creadoo.homer.shell.data.commands;

import java.util.concurrent.TimeUnit;

import org.apache.karaf.shell.api.action.Argument;
import org.apache.karaf.shell.api.action.Command;
import org.apache.karaf.shell.api.action.Completion;
import org.apache.karaf.shell.api.action.lifecycle.Service;
import org.apache.karaf.shell.support.table.Col;
import org.apache.karaf.shell.support.table.Row;
import org.apache.karaf.shell.support.table.ShellTable;

import at.creadoo.homer.processing.data.util.Util;
import at.creadoo.homer.shell.data.commands.completers.FlatCompleter;
import at.creadoo.homer.shell.data.commands.completers.TimeUnitCompleter;

/**
 * Command to display values provided by services.
 */
@Command(scope = "data", name = "display", description = "Show values provided by given services.")
@Service
public class DataDisplay extends CommandSupport {

	@Argument(index = 0, name = "flat", description = "The flat id which should be used.", required = true, multiValued = false)
    @Completion(FlatCompleter.class)
	Integer flatId;

	@Argument(index = 1, name = "period", description = "The desired period.", required = false, multiValued = false)
	Integer period;

	@Argument(index = 2, name = "unit", description = "The desired time unit.", required = false, multiValued = false)
    @Completion(TimeUnitCompleter.class)
	String sUnit;

    @Override
	public Object execute() throws Exception {
    	final ShellTable table = new ShellTable();
    	table.column(new Col("Service").alignLeft());
        
    	final TimeUnit unit;
    	if (sUnit == null || sUnit.isEmpty()) {
    		unit = null;
    	} else {
    		unit = TimeUnit.valueOf(TimeUnit.class, sUnit.trim());
    	}
    	
        if (period != null && period > 0 && unit != null) {
            table.column(new Col("Value [Last " + period + " " + unit.name() + "]").alignLeft());
            
    		if (getHumidityService() != null) {
    			printRow(table, HUMIDITY_SERVICE_TITLE, Util.round(getHumidityService().getValue(flatId, period, unit)));
    		} else {
    			printError(ERROR_HUMIDITY_SERVICE);
    		}
    		if (getIndoorClimateService() != null) {
    			printRow(table, INDOOR_CLIMATE_SERVICE_TITLE, getIndoorClimateService().getIndoorClimateType(flatId, period, unit).toString());
    		} else {
    			printError(ERROR_INDOOR_CLIMATE_SERVICE);
    		}
    		if (getRainService() != null) {
    			final Double value = Util.round(getRainService().getValue(flatId, period, unit));
    			printRow(table, RAIN_SERVICE_TITLE, value + " [" + getRainService().getRainType(value, period, unit) + "]");
    		} else {
    			printError(ERROR_RAIN_SERVICE);
    		}
    		if (getTemperatureInsideService() != null) {
    			printRow(table, TEMPERATURE_INSIDE_SERVICE_TITLE, Util.round(getTemperatureInsideService().getValue(flatId)));
    		} else {
    			printError(ERROR_TEMPERATURE_INSIDE_SERVICE);
    		}
    		if (getTemperatureOutsideService() != null) {
    			printRow(table, TEMPERATURE_OUTSIDE_SERVICE_TITLE, Util.round(getTemperatureOutsideService().getValue(flatId)));
    		} else {
    			printError(ERROR_TEMPERATURE_OUTSIDE_SERVICE);
    		}
        } else {
            table.column(new Col("Last Value").alignLeft());
            
    		if (getHumidityService() != null) {
    			printRow(table, HUMIDITY_SERVICE_TITLE, Util.round(getHumidityService().getLastValue(flatId)));
    		} else {
    			printError(ERROR_HUMIDITY_SERVICE);
    		}
    		if (getIndoorClimateService() != null) {
    			printRow(table, INDOOR_CLIMATE_SERVICE_TITLE, getIndoorClimateService().getIndoorClimateType(flatId).toString());
    		} else {
    			printError(ERROR_INDOOR_CLIMATE_SERVICE);
    		}
    		if (getRainService() != null) {
    			final Double value = Util.round(getRainService().getLastValue(flatId));
    			printRow(table, RAIN_SERVICE_TITLE, value + " [" + getRainService().getRainType(value) + "]");
    		} else {
    			printError(ERROR_RAIN_SERVICE);
    		}
    		if (getTemperatureInsideService() != null) {
    			printRow(table, TEMPERATURE_INSIDE_SERVICE_TITLE, Util.round(getTemperatureInsideService().getLastValue(flatId)));
    		} else {
    			printError(ERROR_TEMPERATURE_INSIDE_SERVICE);
    		}
    		if (getTemperatureOutsideService() != null) {
    			printRow(table, TEMPERATURE_OUTSIDE_SERVICE_TITLE, Util.round(getTemperatureOutsideService().getLastValue(flatId)));
    		} else {
    			printError(ERROR_TEMPERATURE_OUTSIDE_SERVICE);
    		}
        }
		
		table.print(getConsole());
        
        return null;
    }
    
    private Row printRow(final ShellTable table, final String service, final Double value) {
    	if (value != null) {
    		return printRow(table, service, Double.toString(value));
    	} else {
    		return printRow(table, service, "null");
    	}
    }
    
    private Row printRow(final ShellTable table, final String service, final String value) {
    	if (table == null || service == null) {
    		return null;
    	}
    	
    	final Row row = table.addRow();
		row.addContent(service);
		row.addContent(value);
		return row;
    }

}
