package at.creadoo.homer.processing.presence.impl;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Dictionary;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Hashtable;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.karaf.scheduler.ScheduleOptions;
import org.apache.karaf.scheduler.Scheduler;
import org.apache.log4j.Logger;
import org.joda.time.DateTime;
import org.joda.time.LocalTime;
import org.osgi.framework.Bundle;
import org.osgi.framework.BundleContext;
import org.osgi.framework.FrameworkUtil;
import org.osgi.framework.ServiceRegistration;
import org.osgi.service.cm.ConfigurationAdmin;
import org.osgi.service.cm.ConfigurationException;
import org.osgi.service.cm.ManagedService;
import org.osgi.service.event.Event;
import org.osgi.service.event.EventAdmin;
import org.osgi.service.event.EventHandler;

import at.ac.ait.hbs.homer.core.common.DataAccess;
import at.ac.ait.hbs.homer.core.common.configuration.ConfigurationService;
import at.ac.ait.hbs.homer.core.common.configuration.ConfigurationValue;
import at.ac.ait.hbs.homer.core.common.configuration.ConfigurationValueChangeListener;
import at.ac.ait.hbs.homer.core.common.enumerations.DeviceCategoryType;
import at.ac.ait.hbs.homer.core.common.enumerations.DeviceMessageType;
import at.ac.ait.hbs.homer.core.common.event.EventBuilder;
import at.ac.ait.hbs.homer.core.common.event.EventProperties;
import at.ac.ait.hbs.homer.core.common.event.EventTopic;
import at.ac.ait.hbs.homer.core.common.event.util.EventUtil;
import at.ac.ait.hbs.homer.core.common.model.DBDevice;
import at.ac.ait.hbs.homer.core.common.model.DBMessage;
import at.creadoo.homer.processing.presence.Constants;
import at.creadoo.homer.processing.presence.PresenceSimulator;
import at.creadoo.homer.processing.presence.model.ActuatorItem;
import at.creadoo.homer.processing.presence.model.SensorItem;
import at.creadoo.homer.processing.presence.util.RandomUtil;
import at.creadoo.homer.processing.presence.util.Util;
import at.creadoo.homer.processing.presence.util.WeatherUtil;

public class PresenceSimulatorImpl implements PresenceSimulator, ManagedService, EventHandler, ConfigurationValueChangeListener {

	private static final Logger log = Logger.getLogger(PresenceSimulatorImpl.class);

	private static final String SCHEDULER_NAME_RECALCULATE = "recalculation-trigger";

	private final RandomUtil rand = new RandomUtil();

	private ServiceRegistration<?> serviceRegistration;

	private final Map<Integer, DBDevice> actuators = new HashMap<Integer, DBDevice>();
	private final Set<ActuatorItem> actuatorItems = Collections.synchronizedSet(new HashSet<ActuatorItem>());

	private final Map<Integer, DBDevice> sensors = new HashMap<Integer, DBDevice>();
	private final Set<SensorItem> sensorItems = Collections.synchronizedSet(new HashSet<SensorItem>());

	private final Map<String, ActuatorItem> scheduledJobs = new HashMap<String, ActuatorItem>();

	private final Map<String, String> variables = new HashMap<String, String>();

    private ConfigurationAdmin configurationAdmin;

    private ConfigurationService configurationService;
    
    private DataAccess dataAccess;
	
    private EventAdmin eventAdmin;

    private Scheduler scheduler;
    
    private boolean isEnabled = false;
	
    private void setEnabled(final boolean isEnabled) {
    	this.isEnabled = isEnabled;
    	configurationService.setMetaValue(new ConfigurationValue(Constants.CONFIGURATION_KEY_ENABLED, this.isEnabled, false));
    }
    
    private Dictionary<String, ?> getConfiguration() {
    	try {
			return configurationAdmin.getConfiguration(Constants.CONFIG_PID, "?").getProperties();
		} catch (IOException ex) {
			log.error("Error loading configuration", ex);
		}
    	return null;
    }
    
    public void init() {
		log.info("Initialize " + this.getClass().getSimpleName() + "...");
		
		updateDevices();

		configurationService.addListener(this);

		boolean enabled = false;
		final Map<String, String> props = Util.toMap(getConfiguration());
		if (props != null && props.containsKey(Constants.KEY_INITIAL_ENABLED)) {
			final String initialEnabled = Util.getProperty(props, Constants.KEY_INITIAL_ENABLED);
			if (initialEnabled.trim().equalsIgnoreCase("1") || initialEnabled.trim().equalsIgnoreCase("true")) {
				enabled = true;
			}
		}
		
		if (enabled) {
			log.info("Initially enabled");
		} else {
			log.info("Initially disabled");
		}
		setEnabled(enabled);
		
		registerService();
    }

    public void destroy() {
		log.info("Destroy " + this.getClass().getSimpleName() + "...");

    	configurationService.removeListener(this);
		
		unregister();

		unregisterService();
    }

    public final void setConfigurationAdmin(final ConfigurationAdmin configurationAdmin) {
    	this.configurationAdmin = configurationAdmin;
    }
    
    public final void setConfigurationService(final ConfigurationService configurationService) {
    	this.configurationService = configurationService;
    }

	public final void setDataAccess(final DataAccess dataAccess) {
		this.dataAccess = dataAccess;
	}

	public void setEventAdmin(final EventAdmin eventAdmin) {
		this.eventAdmin = eventAdmin;
	}

	public final void setScheduler(final Scheduler scheduler) {
		this.scheduler = scheduler;
	}

	@Override
	public void enable() {
		doEnable();
	}

	@Override
	public void disable() {
		doDisable();
	}

	@Override
	public boolean isEnabled() {
		return isEnabled;
	}

	@Override
	public List<ActuatorItem> getActuatorItems() {
		synchronized (actuatorItems) {
			return new ArrayList<ActuatorItem>(actuatorItems);
		}
	}

	@Override
	public List<SensorItem> getSensorItems() {
		synchronized (sensorItems) {
			return new ArrayList<SensorItem>(sensorItems);
		}
	}
	

	@Override
	public Map<String, String> getVariables() {
		synchronized (variables) {
			return new HashMap<String, String>(variables);
		}
	}
	
	@Override
	public void recalculate() {
		try {
			this.updated(getConfiguration());
		} catch (ConfigurationException ex) {
			log.error("Error while initializing recalculation", ex);
		}
	}
	
	@Override
	public void reset() {
		log.debug("Reset state of managed devices to planned state");

		// Prepare items
		log.debug("Preparing items");
		final Map<DBDevice, Set<ActuatorItem>> itemsPerDevice = new HashMap<DBDevice, Set<ActuatorItem>>();
		synchronized (actuatorItems) {
			for (ActuatorItem item : this.actuatorItems) {
				if (item.getDevice() == null) {
					continue;
				}
				
				if (itemsPerDevice.containsKey(item.getDevice())) {
					itemsPerDevice.get(item.getDevice()).add(item);
				} else {
					final Set<ActuatorItem> items = new HashSet<ActuatorItem>();
					items.add(item);
					itemsPerDevice.put(item.getDevice(), items);
				}
			}
		}

		// Calculate status for every device
		log.debug("Checking devices");
		for (DBDevice device : itemsPerDevice.keySet()) {
			log.debug("Checking device '" + device.getId() + "' with " + itemsPerDevice.get(device).size() + " item(s)");

			final List<DeviceMessageType> deviceMessageTypes = DeviceMessageType.getFlagsForDeviceType(device.getType());
			if (deviceMessageTypes.size() > 2) {
				log.debug("Skip device '" + device.getId() + "'");
				continue;
			}
			
			boolean switched = false;
			ActuatorItem lastItem = null;
			
			for (ActuatorItem item : itemsPerDevice.get(device)) {
				if (item.getStartTime() == null || item.getEndTime() == null) {
					log.error("Start or end time not set for device '" + device.getId() + "'");
					continue;
				}

				lastItem = item;

				// Device should be active
				if (item.getEndTime() != null && LocalTime.now().isAfter(item.getStartTime()) && LocalTime.now().isBefore(item.getEndTime())) {
					log.debug("Set device '" + device.getId() + "' to '" + item.getDeviceMessageType() + "'");
					postEvent(item, item.getDeviceMessageType());
					switched = true;
					break;
				}
			}

			// Device is not active currently
			if (!switched && lastItem != null) {
				final DeviceMessageType messageType = getOpposite(deviceMessageTypes, lastItem.getDeviceMessageType());
				log.debug("Set device '" + device.getId() + "' to '" + messageType + "'");
				postEvent(lastItem, messageType);
			}
		}
	}

    @Override
    public void handleEvent(final Event event) {
		log.debug("Received " + event);
		try {
			if (EventUtil.isEventOfType(event, EventTopic.HOMECONTROL_DEVICE_MESSAGE)) {
				final DBMessage message = (DBMessage) event.getProperty(EventProperties.MESSAGE.name());
				
				for (SensorItem item : sensorItems) {
					if (item.getDevice().getId().equals(message.getDeviceId()) && item.getDeviceMessageType().equals(message.getMessageType())) {
						if (item.getOn() && !this.isEnabled) {
							doEnable();
						} else if (!item.getOn() && this.isEnabled) {
							doDisable();
						}
						return;
					}
				}
			} else if (EventUtil.isEventOfType(event, EventTopic.HOMECONTROL_DEVICES_CONFIGURATION_CHANGE_MESSAGE)) {
				log.debug("Reloading device cache");
				this.updateDevices();
			}
		} catch (Throwable ex) {
			log.error("Error while processing incoming event", ex);
		}
    }

	@Override
	public final void handleValueChange(final String key, final ConfigurationValue valueOld, final ConfigurationValue valueNew) {
		// When value is removed from configuration valueNew == null
		if (valueNew == null) {
			return;
		}
		
		if (key.equals(Constants.CONFIGURATION_KEY_ENABLED)) {
			try {
				boolean enabled = valueNew.getBooleanValue();
				
				if (!this.isEnabled && enabled) {
					log.info("Enabled");
					setEnabled(enabled);
					this.updated(getConfiguration());
				} else if (this.isEnabled && !enabled) {
					log.info("Disabled");
					setEnabled(enabled);
					unregister();
				}
			} catch (Throwable ex) {
				log.error("Error while handling configuration value change for key '" + key + "'", ex);
			}
		}
	}

	@Override
	public synchronized void updated(final Dictionary<String, ?> properties) throws ConfigurationException {
		if (properties != null) {
			updated(Util.toMap(properties));
		}
	}

	public synchronized void updated(final Map<String, String> properties) throws ConfigurationException {
		if (properties != null) {
			unregister();
			
			if (properties.containsKey(Constants.KEY_RECALCULATION_TIME)) {
				final String timeValue = properties.get(Constants.KEY_RECALCULATION_TIME).trim();
				if (timeValue != null && !timeValue.isEmpty()) {
					final LocalTime time = Constants.parseTime(timeValue);
					if (time != null) {
						try {
							DateTime executionTime = DateTime.now().withTime(time);
							if (executionTime.isBefore(DateTime.now())) {
								executionTime = executionTime.plusDays(1);
							}
							
							final ScheduleOptions scheduleOptions = scheduler.AT(executionTime.toDate());
							scheduleOptions.name(SCHEDULER_NAME_RECALCULATE);
							scheduleOptions.canRunConcurrently(false);
							scheduler.schedule(new Runnable() {
								@Override
								public void run() {
									try {
										PresenceSimulatorImpl.this.updated(PresenceSimulatorImpl.this.getConfiguration());
									} catch (ConfigurationException ex) {
										log.error("Error executing recalculation", ex);
									}
								}
							}, scheduleOptions);
						} catch (Exception ex) {
							log.error("Error registering job", ex);
						}
					}
				}
			}
			
			process(properties);
		}
	}
	
	private void updateDevices() {
		final Map<Integer, DBDevice> sensors = dataAccess.getDevicesByDeviceCategoryTypeAsMap(DeviceCategoryType.MDC_AI_TYPE_SENSOR);
		log.debug(sensors != null ? sensors.size() + " sensors found" : "No sensors found!");
		this.sensors.clear();
		if (sensors != null) {
			this.sensors.putAll(sensors);
		}
		
		final Map<Integer, DBDevice> actuators = dataAccess.getDevicesByDeviceCategoryTypeAsMap(DeviceCategoryType.MDC_AI_TYPE_ACTUATOR);
		log.debug(actuators != null ? actuators.size() + " actuators found" : "No actuators found!");
		this.actuators.clear();
		if (actuators != null) {
			this.actuators.putAll(actuators);
		}
	}
	
	private void process(final Map<String, String> properties) {
		if (this.isEnabled) {
			synchronized (variables) {
				log.debug("Requesting variables");
				variables.clear();
				variables.putAll(WeatherUtil.fetchVariables(properties));
				
				log.debug("Replacing variables");
				Util.replaceProperty(properties, "<" + Constants.VARIABLE_KEY_SUNRISE + ">", variables.get(Constants.VARIABLE_KEY_SUNRISE));
				Util.replaceProperty(properties, "<" + Constants.VARIABLE_KEY_SUNSET + ">", variables.get(Constants.VARIABLE_KEY_SUNSET));
			}
			
			synchronized (actuatorItems) {
				actuatorItems.clear();
			}
			
			synchronized (sensorItems) {
				sensorItems.clear();
			}
			
			// Process actuators
			final Map<String, String> actuatorProperties = Util.matchingSubset(properties, Constants.KEY_PREFIX_ACTUATOR, false);

			for (String key : actuatorProperties.keySet()) {
				final String value = Util.prepareProperty(actuatorProperties.get(key));
				final List<String> keyParts = Arrays.asList(key.split("\\."));
				
				if (keyParts != null && keyParts.size() >= 2) {
					int randomizeMinutes = 0;
					if (keyParts.size() >= 3) {
						try {
							randomizeMinutes = Integer.parseInt(keyParts.get(2));
						} catch (NumberFormatException ex) {
							//
						}
					}
					
					try {
						final Integer actuatorId = Integer.parseInt(keyParts.get(0));
						final DeviceMessageType actuatorMessageType = DeviceMessageType.valueOf(keyParts.get(1));
						
						if (actuatorId == null || actuatorMessageType == null) {
							log.error("Unable to parse entry with key '" + Constants.KEY_PREFIX_ACTUATOR + key + "'");
							continue;
						}
					
						log.debug("Actuator id: " + actuatorId);
						log.debug("Actuator message type: " + actuatorMessageType);
						
						if (!actuators.containsKey(actuatorId)) {
							log.error("Unable to find actuator with id '" + actuatorId + "'");
							continue;
						}
						
						final DBDevice actuator = actuators.get(actuatorId);
						if (!DeviceMessageType.getFlagsForDeviceType(actuator.getType()).contains(actuatorMessageType)) {
							log.error("Given DeviceMessageType '" + actuatorMessageType + "' not valid for actuator with id '" + actuatorId + "'");
							continue;
						}
						
						for (String time : value.split(",")) {
							// Get times
							final LocalTime startTime = Util.getStartTime(time);
							final LocalTime endTime = Util.getEndTime(time);
							
							if (startTime == null) {
								log.error("Given time '" + time + "' not parsable");
								continue;
							}
							if (endTime != null && startTime.isEqual(endTime)) {
								log.error("End time shouldn't be the same as start time");
								continue;
							}
	
							final ActuatorItem presenceItem = new ActuatorItem(actuator, actuatorMessageType, startTime, endTime);
							if (randomizeMinutes > 0) {
								randomizeItem(presenceItem, randomizeMinutes);
							}
							
							synchronized (actuatorItems) {
								actuatorItems.add(presenceItem);
							}
						}
					} catch (NumberFormatException ex) {
						log.error("Unable to parse device id", ex);
						continue;
					}
					
					synchronized (actuatorItems) {
						register(actuatorItems);
					}
				} else {
					log.error("Given key '" + key + "' not valid");
				}
			}
		}
		
		// Process sensors
		final Map<String, String> sensorProperties = Util.matchingSubset(properties, Constants.KEY_PREFIX_SENSOR, false);

		synchronized (sensorItems) {
			sensorItems.clear();
		}
		for (String key : sensorProperties.keySet()) {
			log.debug("Processing key '" + key + "'");
			
			final String value = Util.prepareProperty(sensorProperties.get(key));
			final List<String> keyParts = Arrays.asList(key.split("\\."));
			
			if (keyParts != null && keyParts.size() >= 2) {
				try {
					final Integer sensorId = Integer.parseInt(keyParts.get(0));
					final DeviceMessageType sensorMessageType = DeviceMessageType.valueOf(keyParts.get(1));
					
					if (sensorId == null || sensorMessageType == null) {
						log.error("Unable to parse entry with key '" + Constants.KEY_PREFIX_SENSOR + key + "'");
						continue;
					}
				
					log.debug("Sensor id: " + sensorId);
					log.debug("Sensor message type: " + sensorMessageType);
					
					if (!sensors.containsKey(sensorId)) {
						log.error("Unable to find sensor with id '" + sensorId + "'");
						continue;
					}
					
					final DBDevice sensor = sensors.get(sensorId);
					if (!DeviceMessageType.getFlagsForDeviceType(sensor.getType()).contains(sensorMessageType)) {
						log.error("Given DeviceMessageType '" + sensorMessageType + "' not valid for sensor with id '" + sensorId + "'");
						continue;
					}
					
					final SensorItem sensorItem;
					if (value != null && value.trim().equalsIgnoreCase("on")) {
						sensorItem = new SensorItem(sensor, sensorMessageType, true);
					} else {
						sensorItem = new SensorItem(sensor, sensorMessageType, false);
					}
					sensorItems.add(sensorItem);
				} catch (NumberFormatException ex) {
					log.error("Unable to parse device id", ex);
					continue;
				}
			} else {
				log.error("Given key '" + key + "' not valid");
			}
		}
	}
	
	private void register(final Set<ActuatorItem> presenceItems) {
		/** TODO:
		 * - Check if events/intervals overlap and modify or ignore the affected events to avoid unwanted behavior
		 * - Do this per sensor - take into account all items that refer to a certain sensor regardless of message type
		 * 
		**/
		for (ActuatorItem item : presenceItems) {
			final List<DeviceMessageType> deviceMessageTypes = DeviceMessageType.getFlagsForDeviceType(item.getDevice().getType());
			if (item.getEndTime() != null && deviceMessageTypes.size() > 2) {
				log.error("End time not possible for device '" + item.getDevice().getId() + "' which has " + deviceMessageTypes.size() + " possible DeviceMessageTypes");
				continue;
			}
			
			if (item.getEndTime() == null) {
				// Schedule only start time
				schedulePresenceItem("actuator-" + item.getDevice().getId() + "-start-" + item.getStartTime(), item, item.getDeviceMessageType(), item.getStartTime());
			} else if (item.getEndTime() != null && LocalTime.now().isAfter(item.getStartTime()) && LocalTime.now().isBefore(item.getEndTime())) {
				// Set actuator now and schedule end time
				postEvent(item, item.getDeviceMessageType());
				schedulePresenceItem("actuator-" + item.getDevice().getId() + "-end-" + item.getEndTime(), item, getOpposite(deviceMessageTypes, item.getDeviceMessageType()), item.getEndTime());
			} else {
				// Schedule start and end time
				schedulePresenceItem("actuator-" + item.getDevice().getId() + "-start-" + item.getStartTime(), item, item.getDeviceMessageType(), item.getStartTime());
				schedulePresenceItem("actuator-" + item.getDevice().getId() + "-end-" + item.getEndTime(), item, getOpposite(deviceMessageTypes, item.getDeviceMessageType()), item.getEndTime());
			}
		}
	}
	
	private void randomizeItem(final ActuatorItem item, final Integer randomMinutes) {
		item.setStartTime(item.getStartTime().plusMinutes(rand.nextInt(-1 * randomMinutes, randomMinutes)));
		
		if (item.getEndTime() != null) {
			final LocalTime randomEndTime = item.getEndTime().plusMinutes(rand.nextInt(-1 * randomMinutes, randomMinutes));
			if (randomEndTime.isBefore(item.getStartTime())) {
				item.setEndTime(item.getStartTime().plusMinutes(1));
			} else {
				item.setEndTime(randomEndTime);
			}
		}
	}
	
	private DeviceMessageType getOpposite(final List<DeviceMessageType> deviceMessageTypes, final DeviceMessageType deviceMessageType) {
		if (deviceMessageTypes.size() == 2) {
			for (DeviceMessageType item : deviceMessageTypes) {
				if (!item.equals(deviceMessageType)) {
					return item;
				}
			}
		}
		return null;
	}
	
	private synchronized void schedulePresenceItem(final String name, final ActuatorItem item, final DeviceMessageType messageType, final LocalTime time) {
		if (name == null || name.isEmpty() || item == null || messageType == null || time == null) {
			return;
		}
		
		DateTime executionTime = time.toDateTimeToday();
		if (executionTime.isBefore(DateTime.now())) {
			executionTime = executionTime.plusDays(1);
		}
		
		log.debug("schedulePresenceItem: " + time + " -> " + name + " (" + executionTime + ")");
		
		final ScheduleOptions scheduleOptions = scheduler.AT(executionTime.toDateTime().toDate(), -1, Constants.SECONDS_PER_DAY);
		scheduleOptions.name(name);
		scheduleOptions.canRunConcurrently(false);
		
		try {
			scheduler.schedule(new Runnable() {
				@Override
				public void run() {
					postEvent(item, messageType);
				}
			}, scheduleOptions);

			synchronized (scheduledJobs) {
				scheduledJobs.put(name, item);
			}
		} catch (Throwable ex) {
			log.error("Error scheduling item", ex);
		}
	}
	
	private void postEvent(final ActuatorItem item, final DeviceMessageType messageType) {
		final Event event = EventBuilder.createDeviceRequestEvent(item.getDevice(), messageType);
		log.info("Send device request for device '" + item.getDevice().getId() + "' and message type '" + messageType.name() + "'");
		eventAdmin.postEvent(event);
	}
    
    private void doEnable() {
    	if (this.isEnabled) {
    		return;
    	}
    	
		try {
			// reload config & start up
			log.info("Enabled");
			setEnabled(true);
			this.updated(getConfiguration());
		} catch (Throwable ex) {
			log.error("Error while enabling", ex);
		}
    }
    
    private void doDisable() {
    	if (!this.isEnabled) {
    		return;
    	}
    	
		try {
			// disable service
			log.info("Disabled");
			setEnabled(false);
			unregister();
		} catch (Throwable ex) {
			log.error("Error while enabling", ex);
		}
    }
	
	private synchronized void unregister() {
		scheduler.unschedule(SCHEDULER_NAME_RECALCULATE);
		
		synchronized (scheduledJobs) {
			for (String name : scheduledJobs.keySet()) {
				scheduler.unschedule(name);
			}
			scheduledJobs.clear();
		}
	}

	private void registerService() {
		final Bundle bundle = FrameworkUtil.getBundle(this.getClass());
		if (bundle == null) {
			return;
		}
		
		final BundleContext bundleContext = bundle.getBundleContext();
		if (bundleContext == null) {
			return;
		}
		
		// Register configuration change listener
		final Hashtable<String, Object> properties = new Hashtable<String, Object>();
		properties.put(org.osgi.framework.Constants.SERVICE_PID, Constants.CONFIG_PID);
		serviceRegistration = bundleContext.registerService(ManagedService.class.getName(), this, properties);
	}
	
	private void unregisterService() {
		if (serviceRegistration != null) {
			serviceRegistration.unregister();
		}
	}

}