package at.creadoo.homer.processing.presence.util;

import java.math.BigDecimal;
import java.util.Dictionary;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.apache.log4j.Logger;
import org.joda.time.DateTimeFieldType;
import org.joda.time.LocalTime;

import com.udojava.evalex.Expression;

import at.ac.ait.hbs.homer.core.common.util.PropertyUtil;
import at.creadoo.homer.processing.presence.Constants;

public class Util {

	private static final Logger log = Logger.getLogger(Util.class);

	public Util() {
		//
	}

	public static Map<String, String> toMap(final Dictionary<String, ?> dictionary) {
		if (dictionary == null) {
			return new HashMap<String, String>();
		}
		final Map<String, String> map = new HashMap<String, String>(dictionary.size());
		Enumeration<String> keys = dictionary.keys();
		while (keys.hasMoreElements()) {
			String key = keys.nextElement();
			map.put(key, dictionary.get(key).toString());
		}
		return map;
	}

	/**
	 * Extracts a specific property key subset from the map passed. The
	 * prefix may be removed from the keys in the resulting map, or it
	 * may be kept. In the latter case, exact matches on the prefix will also be
	 * copied into the resulting map.
	 *
	 *
	 * @param prefix
	 *            is the key prefix to filter the map keys by.
	 * @param keepPrefix
	 *            if true, the key prefix is kept in the resulting map.
	 *            As side-effect, a key that matches the prefix exactly will
	 *            also be copied. If false, the resulting dictionary's keys are
	 *            shortened by the prefix. An exact prefix match will not be
	 *            copied, as it would result in an empty string key.
	 * @return a map matching the filter key. May be an empty
	 *         map, if no prefix matches were found.
	 *
	 * @see Map#get(String) Map.get() is used to assemble matches
	 */
	public static Map<String, String> matchingSubset(final Map<String, String> properties, final String prefix, final boolean keepPrefix) {
		final Map<String, String> result = new HashMap<String, String>();

		// sanity check
		if (prefix == null || prefix.length() == 0) {
			return result;
		}

		String prefixMatch; // match prefix strings with this
		String prefixSelf; // match self with this
		if (prefix.charAt(prefix.length() - 1) != '.') {
			// prefix does not end in a dot
			prefixSelf = prefix;
			prefixMatch = prefix + '.';
		} else {
			// prefix does end in one dot, remove for exact matches
			prefixSelf = prefix.substring(0, prefix.length() - 1);
			prefixMatch = prefix;
		}
		// POSTCONDITION: prefixMatch and prefixSelf are initialized!

		// now add all matches into the resulting properties.
		// Remark 1: #propertyNames() will contain the System properties!
		// Remark 2: We need to give priority to System properties. This is done
		// automatically by calling this class's getProperty method.
		for (String key : properties.keySet()) {
			if (keepPrefix) {
				// keep full prefix in result, also copy direct matches
				if (key.startsWith(prefixMatch) || key.equals(prefixSelf)) {
					result.put(key, properties.get(key));
				}
			} else {
				// remove full prefix in result, don't copy direct matches
				if (key.startsWith(prefixMatch)) {
					result.put(key.substring(prefixMatch.length()), properties.get(key));
				}
			}
		}

		// done
		return result;
	}

	public static String getProperty(final Map<String, String> properties, final String key) {
    	log.debug("Get value for '" + key + "' from: " + properties);
    	if (properties != null && properties.keySet().contains(key)) {
    		return PropertyUtil.sanitizeValue(properties.get(key));
    	}
    	return "";
    }

	public static void replaceProperty(final Map<String, String> properties, final String target, final String replacement) {
		for (String key : properties.keySet()) {
			final String item = properties.get(key);
			if (item.contains(target)) {
				properties.put(key, item.replace(target, replacement));
			}
		}
	}

	public static String prepareProperty(final String property) {
		if (property == null || property.trim().isEmpty()) {
			return "";
		}
		
		String temp = property.trim();
		
		// Replace spaces
		temp = temp.trim().replace(" ", "");

		// Search for expressions and evaluate
		String valueToReplace = Util.getContentsInBrackets(temp);
		while (valueToReplace != null) {
			try {
				final Expression expr = new Expression(Util.convertTimesToInt(valueToReplace));
				final BigDecimal evaluatedValueToReplace = expr.eval();
				
				final String replacement = Util.printTimeFromInt(evaluatedValueToReplace);
				temp = temp.replace(valueToReplace, replacement);
				valueToReplace = Util.getContentsInBrackets(temp, temp.indexOf(replacement) + replacement.length());
			} catch (Throwable ex) {
				log.error("Error evaluating expression '" + valueToReplace + "'");
				valueToReplace = Util.getContentsInBrackets(temp, temp.indexOf(valueToReplace) + valueToReplace.length());
			}
		}
		
		return temp;
	}

	private static int getMinutesSinceMidnight(final LocalTime time) {
		return time.get(DateTimeFieldType.minuteOfDay());
	}

	private static String printTimeFromInt(final BigDecimal value) {
		return printTimeFromInt(value.intValue());
	}

	private static String printTimeFromInt(final int value) {
		return Constants.TIME_FORMAT.print(Util.timeFromInt(value));
	}

	private static LocalTime timeFromInt(final int minutesSinceMidnight) {
		return LocalTime.fromMillisOfDay(minutesSinceMidnight * 60 * 1000);
	}

	private static String getContentsInBrackets(final String value) {
		return getContentsInBrackets(value, 0);
	}

	private static String getContentsInBrackets(final String value, final int fromIndex) {
		if (value == null || value.trim().isEmpty()) {
			return null;
		}
		
		final int openPos = value.trim().indexOf("(", fromIndex);
		if (openPos >= 0) {
			final String valueTemp = value.trim();
			int openCounter = 1;
			for (int i = openPos + 1; i < valueTemp.length(); i++) {
		        if (valueTemp.charAt(i) == '(') {
		        	openCounter++;
		        } else if (valueTemp.charAt(i) == ')') {
		        	openCounter--;
		        }
		        
		        if (openCounter == 0) {
		        	return value.trim().substring(openPos, i + 1);
		        }
		    }
		}
		
		return null;
	}

	private static String convertTimesToInt(final String value) {
		final Pattern pattern = Pattern.compile("(\\d\\d:\\d\\d)");
		final Matcher matcher = pattern.matcher(value);
		
		final StringBuffer result = new StringBuffer();
		while (matcher.find()) {
			
			final LocalTime time = Constants.parseTime(matcher.group());
			if (time != null) {
				matcher.appendReplacement(result, Integer.toString(getMinutesSinceMidnight(time)));
			} else {
				matcher.appendReplacement(result, matcher.group());
			}
		}
		matcher.appendTail(result);

		return result.toString();
	}

	public static LocalTime getStartTime(final String time) {
		if (time == null || time.trim().isEmpty()) {
			return null;
		}

		String tempTime = time.trim();
		if (tempTime.contains("+")) {
			tempTime = tempTime.substring(0, time.indexOf("+"));
		} else if (time.contains("-")) {
			tempTime = tempTime.substring(0, time.indexOf("-"));
		}
		
		try {
			return Constants.parseTime(tempTime);
		} catch (Throwable ex) {
			return null;
		}
	}
	
	public static LocalTime getEndTime(final String time) {
		final LocalTime startTime = getStartTime(time);
		
		if (time == null || time.isEmpty()) {
			return null;
		}
		
		if (time.contains("+") && startTime != null) {
			return startTime.plusMinutes(Integer.parseInt(time.substring(time.indexOf("+") + 1)));
		} else if (time.contains("-")) {
			try {
				return Constants.parseTime(time.substring(time.indexOf("-") + 1));
			} catch (Throwable ex) {
				//
			}
		}
		
		return null;
	}

}
