package at.creadoo.homer.processing.data.util;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.concurrent.TimeUnit;

import org.apache.commons.math3.stat.descriptive.DescriptiveStatistics;
import org.joda.time.DateTime;

import at.ac.ait.hbs.homer.core.common.DataAccess;
import at.ac.ait.hbs.homer.core.common.enumerations.DeviceType;
import at.ac.ait.hbs.homer.core.common.model.DBDevice;
import at.creadoo.homer.processing.data.enumerations.AverageCalculationMode;

public class Util {

	private Util() {
		//
	}

	public static final List<Integer> getDeviceIds(final DataAccess dataAccess, final DeviceType deviceType, final int flatId) {
		return new ArrayList<Integer>(dataAccess.getDevicesByFlatIdAndDeviceTypeAsMap(flatId, deviceType).keySet());
	}

	public static final List<Integer> getDeviceIds(final DataAccess dataAccess, final DeviceType deviceType, final int flatId, final List<Integer> excludedDeviceIds) {
		final List<Integer> result = new ArrayList<Integer>(dataAccess.getDevicesByFlatIdAndDeviceTypeAsMap(flatId, deviceType).keySet());
		if (excludedDeviceIds != null) {
			result.removeAll(excludedDeviceIds);
		}
		return result;
	}

	public static final List<Integer> getDeviceIds(final DataAccess dataAccess, final DeviceType deviceType, final int flatId, final int... excludedDeviceIds) {
		final List<Integer> result = new ArrayList<Integer>(dataAccess.getDevicesByFlatIdAndDeviceTypeAsMap(flatId, deviceType).keySet());
		if (excludedDeviceIds != null) {
			result.removeAll(getDeviceIds(excludedDeviceIds));
		}
		return result;
	}

	public static final List<Integer> getDeviceIds(final Collection<DBDevice> devices) {
		final List<Integer> result = new ArrayList<Integer>();
		for (DBDevice device : devices) {
			result.add(device.getId());
		}
		return result;
	}

	public static final List<Integer> getDeviceIds(final List<DBDevice> devices) {
		return getDeviceIds((Collection<DBDevice>) devices);
	}

	public static final List<Integer> getDeviceIds(final int... devices) {
		final List<Integer> result = new ArrayList<Integer>();
		for (int device : devices) {
			result.add(device);
		}
		return result;
	}
	
	public static final DateTime getTimestampForPeriod(final long duration, final TimeUnit timeUnit) {
		if (timeUnit == null) {
			return null;
		}
		return DateTime.now().minusMillis(Long.valueOf(timeUnit.toMillis(duration)).intValue());
	}

	/**
     * Calculate the average for the given set of data and given mode
     * 
     * @param values
     * @param mode
     * @return
     */
    public static final Double average(final Collection<Double> values, final AverageCalculationMode mode) {
    	switch (mode) {
    	case MEAN:
    		return mean(values);
    	case MEDIAN:
    		return median(values);
    	}
    	return Double.NaN;
	}

	/**
     * Calculate the mean for the given set of data
     * 
     * @param values
     * @return
     */
	public static final double mean(final Collection<Double> values) {
		// Get a DescriptiveStatistics instance
		final DescriptiveStatistics stats = new DescriptiveStatistics();
		// Add the data from the array
		if (!values.isEmpty()) {
			for (Double mark : values) {
				stats.addValue(mark);
			}
		}
		return stats.getMean();
	}

	/**
     * Calculate the median for the given set of data
     * 
     * @param values
     * @return
     */
    public static final double median(final Collection<Double> values) {
		// Get a DescriptiveStatistics instance
		final DescriptiveStatistics stats = new DescriptiveStatistics();
		// Add the data from the array
		if (!values.isEmpty()) {
			for (Double mark : values) {
				stats.addValue(mark);
			}
		}
		return stats.getPercentile(50);
	}
	
	/**
     * Round to 2 decimal places
     * 
     * @param f
     * @return
     */
    public static Float round(final Float f) {
        return round(f, 2);
    }
	
	/**
     * Round to certain number of decimals
     * 
     * @param f
     * @param decimalPlace
     * @return
     */
    public static Float round(final Float f, int decimalPlace) {
    	if (f == null || Float.isNaN(f)) {
    		return Float.NaN;
    	}
        BigDecimal bd = new BigDecimal(Float.toString(f));
        bd = bd.setScale(decimalPlace, BigDecimal.ROUND_HALF_UP);
        return bd.floatValue();
    }
	
	/**
     * Round to 2 decimal places
     * 
     * @param d
     * @return
     */
    public static Double round(final Double d) {
        return round(d, 2);
    }
	
	/**
     * Round to certain number of decimals
     * 
     * @param d
     * @param decimalPlace
     * @return
     */
    public static Double round(final Double d, final int decimalPlace) {
    	if (d == null || Double.isNaN(d)) {
    		return Double.NaN;
    	}
        BigDecimal bd = new BigDecimal(Double.toString(d));
        bd = bd.setScale(decimalPlace, BigDecimal.ROUND_HALF_UP);
        return bd.doubleValue();
    }
}
