package at.creadoo.homer.processing.data.impl;

import java.util.List;
import java.util.concurrent.TimeUnit;

import org.joda.time.DateTime;

import at.creadoo.homer.processing.data.enumerations.AverageCalculationMode;
import at.creadoo.homer.processing.data.enumerations.CalculationMode;
import at.creadoo.homer.processing.data.listeners.ValueChangeListener;

final class ValueChangeListenerItem<T> {

	private final ValueChangeListener<T> listener;
	
	private final CalculationMode calculationMode;
	
	private final AverageCalculationMode averageCalculationMode;
	
	private final Integer flatId;
	
	private final List<Integer> deviceIds;
	
	private final List<Integer> excludedDeviceIds;
	
	private final Long period;
	
	private final TimeUnit timeUnit;
	
	private T currentValue;
	
	private T previousValue;
	
	private DateTime lastUpdate;

	ValueChangeListenerItem(final ValueChangeListener<T> listener, final CalculationMode calculationMethod, final AverageCalculationMode averageCalculationMode, final int flatId) {
		this.listener = listener;
		this.calculationMode = calculationMethod;
		this.averageCalculationMode = averageCalculationMode;
		this.flatId = flatId;
		this.deviceIds = null;
		this.excludedDeviceIds = null;
		this.period = null;
		this.timeUnit = null;
	}

	ValueChangeListenerItem(final ValueChangeListener<T> listener, final CalculationMode calculationMethod, final AverageCalculationMode averageCalculationMode, final int flatId, final List<Integer> deviceIds, final List<Integer> excludedDeviceIds) {
		this.listener = listener;
		this.calculationMode = calculationMethod;
		this.averageCalculationMode = averageCalculationMode;
		this.flatId = flatId;
		this.deviceIds = deviceIds;
		this.excludedDeviceIds = excludedDeviceIds;
		this.period = null;
		this.timeUnit = null;
	}
	
	ValueChangeListenerItem(final ValueChangeListener<T> listener, final CalculationMode calculationMethod, final AverageCalculationMode averageCalculationMode, final int flatId, final List<Integer> deviceIds, final List<Integer> excludedDeviceIds, final long period, final TimeUnit timeUnit) {
		this.listener = listener;
		this.calculationMode = calculationMethod;
		this.averageCalculationMode = averageCalculationMode;
		this.flatId = flatId;
		this.deviceIds = deviceIds;
		this.excludedDeviceIds = excludedDeviceIds;
		this.period = period;
		this.timeUnit = timeUnit;
	}
	
	ValueChangeListener<T> getListener() {
		return listener;
	}
	
	CalculationMode getCalculationMode() {
		return calculationMode;
	}
	
	AverageCalculationMode getAverageCalculationMode() {
		return averageCalculationMode;
	}
	
	Integer getFlatId() {
		return flatId;
	}

	boolean isDeviceIdsSet() {
		return (deviceIds != null && !deviceIds.isEmpty());
	}
	
	boolean hasDeviceId(final Integer deviceId) {
		if (deviceId != null && isDeviceIdsSet()) {
			return getDeviceIds().contains(deviceId);
		}
		return false;
	}
	
	List<Integer> getDeviceIds() {
		return deviceIds;
	}

	boolean isExcludedDeviceIdsSet() {
		return (excludedDeviceIds != null && !excludedDeviceIds.isEmpty());
	}
	
	List<Integer> getExcludedDeviceIds() {
		return excludedDeviceIds;
	}
	
	Long getPeriod() {
		return period;
	}

	TimeUnit getTimeUnit() {
		return timeUnit;
	}

	boolean isCurrentValueSet() {
		return (currentValue != null);
	}

	T getCurrentValue() {
		return currentValue;
	}

	void setCurrentValue(final T currentValue) {
		this.currentValue = currentValue;
	}

	boolean isPreviousValueSet() {
		return (previousValue != null);
	}

	T getPreviousValue() {
		return previousValue;
	}

	void setPreviousValue(final T previousValue) {
		this.previousValue = previousValue;
	}
	
	DateTime getLastUpdate() {
		return lastUpdate;
	}

	void setValue(final T value) {
		if (isCurrentValueSet()) {
			setPreviousValue(currentValue);
			setCurrentValue(value);
			
			if (isCurrentValueSet()) {
				if (!currentValue.equals(previousValue)) {
					onValueChangedAsync(currentValue, previousValue);
				}
			}
		} else {
			setCurrentValue(value);
			onValueChangedAsync(currentValue, previousValue);
		}
		lastUpdate = DateTime.now();
	}
	
	private void onValueChangedAsync(final T currentValue, final T previousValue) {
		if (listener == null || currentValue == null) {
			return;
		}
		
		new Thread(new Runnable() {
			@Override
			public void run() {
				listener.onValueChanged(currentValue, previousValue, calculationMode, flatId);
			}
		}).start();
	}
	
}
