package at.creadoo.homer.processing.data.impl;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;

import org.apache.log4j.Logger;

import at.ac.ait.hbs.homer.core.common.enumerations.DeviceMessageType;
import at.ac.ait.hbs.homer.core.common.enumerations.DeviceType;
import at.ac.ait.hbs.homer.core.common.model.DBMessage;
import at.creadoo.homer.processing.data.Constants;
import at.creadoo.homer.processing.data.RainService;
import at.creadoo.homer.processing.data.enumerations.AverageCalculationMode;
import at.creadoo.homer.processing.data.enumerations.RainType;
import at.creadoo.homer.processing.data.util.Util;

public class RainServiceImpl extends BasicServiceImpl implements RainService {

	private static final Logger log = Logger.getLogger(RainServiceImpl.class);
	
	public RainServiceImpl() {
		super(DeviceType.MDC_AI_TYPE_SENSOR_WEATHER_RAIN, DeviceMessageType.MDC_AI_FLAG_SENSOR_WEATHER_RAIN_READING);
	}
    
    public void init() {
		log.info("Initialize " + this.getClass().getSimpleName() + "...");
    }

    public void destroy() {
		log.info("Destroy " + this.getClass().getSimpleName() + "...");
    }

	@Override
	public Double getValue(final int flatId, final AverageCalculationMode mode, final long calculationPeriod, final TimeUnit timeUnit, final List<Integer> excludedDevices) {
		final List<Integer> deviceIds = getDeviceIds(flatId, excludedDevices);
		if (deviceIds == null || deviceIds.isEmpty()) {
			return Double.NaN;
		}
		
		log.debug("getValue: devices: " + deviceIds);
		
		final Map<Integer, Double> deviceAmounts = new HashMap<Integer, Double>();

		for (Integer deviceId : deviceIds) {
			final List<DBMessage> messages = super.getMessagesForPeriod(calculationPeriod, timeUnit, deviceId);
			
			if (messages != null && messages.size() > 0) {
				double deviceMin = Double.POSITIVE_INFINITY;
				double deviceMax = 0d;

				for (DBMessage message : messages) {
					if (message.getData2() >= 0) {
						deviceMin = Math.min(deviceMin, message.getData2());
						deviceMax = Math.max(deviceMax, message.getData2());
					}
				}

				deviceAmounts.put(deviceId, Math.abs(deviceMax - deviceMin));
			}
		}
		
		log.debug("getValue: amounts: " + deviceAmounts);
		
		return Util.mean(deviceAmounts.values()); 
	}

	@Override
	public RainType getRainType(final int flatId) {
		return getRainType(super.getValue(flatId, Constants.DEFAULT_MODE), Constants.DEFAULT_PERIOD, Constants.DEFAULT_UNIT);
	}

	@Override
	public RainType getRainType(final int flatId, final List<Integer> excludedDevices) {
		return getRainType(super.getValue(flatId, Constants.DEFAULT_MODE, excludedDevices), Constants.DEFAULT_PERIOD, Constants.DEFAULT_UNIT);
	}

	@Override
	public RainType getRainType(final int flatId, final int... excludedDeviceIds) {
		return getRainType(super.getValue(flatId, Constants.DEFAULT_MODE, excludedDeviceIds), Constants.DEFAULT_PERIOD, Constants.DEFAULT_UNIT);
	}

	@Override
	public RainType getRainType(final int flatId, final long calculationPeriod, final TimeUnit timeUnit) {
		return getRainType(super.getValue(flatId, Constants.DEFAULT_MODE, calculationPeriod, timeUnit), calculationPeriod, timeUnit);
	}

	@Override
	public RainType getRainType(final int flatId, final long calculationPeriod, final TimeUnit timeUnit, final int... excludedDeviceIds) {
		return getRainType(super.getValue(flatId, Constants.DEFAULT_MODE, calculationPeriod, timeUnit, excludedDeviceIds), calculationPeriod, timeUnit);
	}

	@Override
	public RainType getRainType(final int flatId, final long calculationPeriod, final TimeUnit timeUnit, List<Integer> excludedDeviceIds) {
		return getRainType(super.getValue(flatId, Constants.DEFAULT_MODE, calculationPeriod, timeUnit, excludedDeviceIds), calculationPeriod, timeUnit);
	}

	@Override
	public RainType getRainType(final int flatId, final AverageCalculationMode mode) {
		return getRainType(super.getValue(flatId, mode), Constants.DEFAULT_PERIOD, Constants.DEFAULT_UNIT);
	}

	@Override
	public RainType getRainType(final int flatId, final AverageCalculationMode mode, final List<Integer> excludedDevices) {
		return getRainType(super.getValue(flatId, mode, excludedDevices), Constants.DEFAULT_PERIOD, Constants.DEFAULT_UNIT);
	}

	@Override
	public RainType getRainType(final int flatId, final AverageCalculationMode mode, final int... excludedDeviceIds) {
		return getRainType(super.getValue(flatId, mode, excludedDeviceIds), Constants.DEFAULT_PERIOD, Constants.DEFAULT_UNIT);
	}

	@Override
	public RainType getRainType(final int flatId, final AverageCalculationMode mode, final long calculationPeriod, final TimeUnit timeUnit) {
		return getRainType(super.getValue(flatId, mode, calculationPeriod, timeUnit), calculationPeriod, timeUnit);
	}

	@Override
	public RainType getRainType(final int flatId, final AverageCalculationMode mode, final long calculationPeriod, final TimeUnit timeUnit, final int... excludedDeviceIds) {
		return getRainType(super.getValue(flatId, mode, calculationPeriod, timeUnit, excludedDeviceIds), calculationPeriod, timeUnit);
	}

	@Override
	public RainType getRainType(final int flatId, final AverageCalculationMode mode, final long calculationPeriod, final TimeUnit timeUnit, List<Integer> excludedDeviceIds) {
		return getRainType(super.getValue(flatId, mode, calculationPeriod, timeUnit, excludedDeviceIds), calculationPeriod, timeUnit);
	}
	
	@Override
	public final RainType getRainType(final Double value) {
		return getRainType(value, Constants.DEFAULT_PERIOD, Constants.DEFAULT_UNIT);
	}
	
	@Override
	public final RainType getRainType(final Double value, final long calculationPeriod, final TimeUnit timeUnit) {
		if (value == null || timeUnit == null) {
			return RainType.UNKNOWN;
		}
		
		// Get the factor to calculate based on an hour
    	final double factor = (double) TimeUnit.HOURS.toMillis(1) / (double) timeUnit.toMillis(calculationPeriod);
		final double valuePerHour = value * factor;
		
		log.debug("Amount projected to an hour (value: " + value + ", period: " + calculationPeriod + ", unit: " + timeUnit + ", factor: " + factor + "): " + valuePerHour);
    	
		if (valuePerHour > 0d && valuePerHour < 2.5d) {
			return RainType.LIGHT;
		} else if (valuePerHour >= 2.5d && valuePerHour < 10d) {
			return RainType.MODERATE;
		} else if (valuePerHour >= 10d && valuePerHour < 50d) {
			return RainType.HEAVY;
		} else if (valuePerHour >= 50d) {
			return RainType.VIOLENT;
		} else {
			return RainType.NONE;
		}
	}

}