package at.creadoo.homer.processing.data.impl;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;

import org.apache.log4j.Logger;
import org.osgi.service.event.Event;
import org.osgi.service.event.EventHandler;

import at.ac.ait.hbs.homer.core.common.DataAccess;
import at.ac.ait.hbs.homer.core.common.event.EventTopic;
import at.ac.ait.hbs.homer.core.common.event.util.EventUtil;
import at.ac.ait.hbs.homer.core.common.model.DBFlat;
import at.creadoo.homer.processing.data.Constants;
import at.creadoo.homer.processing.data.HumidityService;
import at.creadoo.homer.processing.data.IndoorClimateService;
import at.creadoo.homer.processing.data.TemperatureInsideService;
import at.creadoo.homer.processing.data.enumerations.AverageCalculationMode;
import at.creadoo.homer.processing.data.enumerations.CalculationMode;
import at.creadoo.homer.processing.data.enumerations.IndoorClimateType;
import at.creadoo.homer.processing.data.listeners.ValueChangeListener;

public final class IndoorClimateServiceImpl implements IndoorClimateService, ValueChangeListener<Double>, EventHandler {

	private static final Logger log = Logger.getLogger(IndoorClimateServiceImpl.class);
	
    private DataAccess dataAccess;
	
    private HumidityService humidityService;
	
    private TemperatureInsideService temperatureInsideService;
    
    private final Map<Integer, DBFlat> flatCache = new HashMap<Integer, DBFlat>();
    
    private List<ValueChangeListenerItem<IndoorClimateType>> listeners = new ArrayList<ValueChangeListenerItem<IndoorClimateType>>();
	
    public IndoorClimateServiceImpl() {
    	//
    }
    
    public void init() {
		log.info("Initialize " + this.getClass().getSimpleName() + "...");
    	updateFlatCache();
		addListeners();
    }

    public void destroy() {
		log.info("Destroy " + this.getClass().getSimpleName() + "...");
		removeListeners();
    }
    
	public void setDataAccess(final DataAccess dataAccess) {
		this.dataAccess = dataAccess;
	}
    
	public final void setHumidityService(final HumidityService humidityService) {
		this.humidityService = humidityService;
	}
    
	public final void setTemperatureInsideService(final TemperatureInsideService temperatureInsideService) {
		this.temperatureInsideService = temperatureInsideService;
	}

    @Override
    public void handleEvent(final Event event) {
		try {
			if (EventUtil.isEventOfType(event, EventTopic.HOMECONTROL_DEVICES_CONFIGURATION_CHANGE_MESSAGE)) {
				log.debug("Reloading flat cache");
				this.removeListeners();
				this.updateFlatCache();
				this.addListeners();
			}
		} catch (Throwable ex) {
			log.error("Error while processing incoming event", ex);
		}
    }
    
    @Override
	public final void addListenerForLastValue(final ValueChangeListener<IndoorClimateType> listener, final AverageCalculationMode mode, final int flatId) {
		synchronized (listeners) {
			if (!ListenerUtil.containsListener(listeners, listener)) {
				listeners.add(new ValueChangeListenerItem<IndoorClimateType>(listener, CalculationMode.LAST, mode, flatId));
			}
		}
	}

	@Override
	public final void removeListenerForLastValue(final ValueChangeListener<IndoorClimateType> listener) {
		ListenerUtil.removeListener(listeners, listener, CalculationMode.LAST);
	}

	@Override
	public final void removeListener(final ValueChangeListener<IndoorClimateType> listener) {
		ListenerUtil.removeListener(listeners, listener);
	}
	
	@Override
	public final IndoorClimateType getIndoorClimateType(final int flatId) {
		return getIndoorClimateType(flatId, Constants.DEFAULT_MODE);
	}
	
	@Override
	public final IndoorClimateType getIndoorClimateType(final int flatId, final AverageCalculationMode mode) {
		final Double humidityValue = humidityService.getLastValue(flatId);
		final Double temperatureValue = temperatureInsideService.getLastValue(flatId);
		
		return getIndoorClimateType(humidityValue, temperatureValue);
	}
	
	@Override
	public final IndoorClimateType getIndoorClimateType(final int flatId, final long calculationPeriod, final TimeUnit timeUnit) {
		return getIndoorClimateType(flatId, Constants.DEFAULT_MODE, calculationPeriod, timeUnit);
	}
	
	@Override
	public final IndoorClimateType getIndoorClimateType(final int flatId, final AverageCalculationMode mode, final long calculationPeriod, final TimeUnit timeUnit) {
		final Double humidityValue = humidityService.getValue(flatId, mode, calculationPeriod, timeUnit);
		final Double temperatureValue = temperatureInsideService.getValue(flatId, mode, calculationPeriod, timeUnit);
		
		return getIndoorClimateType(humidityValue, temperatureValue);
	}
	
	private void addListeners() {
		synchronized (flatCache) {
			for (Integer flatId : flatCache.keySet()) {
				humidityService.addListenerForLastValue(this, AverageCalculationMode.MEAN, flatId);
				temperatureInsideService.addListenerForLastValue(this, AverageCalculationMode.MEAN, flatId);
			}
		}
	}
	
	private void removeListeners() {
		humidityService.removeListenerForLastValue(this);
		temperatureInsideService.removeListenerForLastValue(this);
	}
	
	private final IndoorClimateType getIndoorClimateType(final Double humidityValue, final Double temperatureValue) {
		if (humidityValue == null || temperatureValue == null) {
			return IndoorClimateType.UNKNOWN;
		}
		
		if (temperatureValue >= 10 && temperatureValue <= 25 && humidityValue >= 40 && temperatureValue <= 70) {
			return IndoorClimateType.COMFORTABLE;
		} else if (humidityValue > 70) {
			return IndoorClimateType.WET;
		} else if (humidityValue < 40) {
			return IndoorClimateType.DRY;
		} else {
			return IndoorClimateType.UNKNOWN;
		}
	}

	@Override
	public void onValueChanged(final Double value, final Double previousValue, final CalculationMode mode, final Integer flatId) {
		new Thread(new Runnable() {
			@Override
			public void run() {
				recalculateIfNecessary(flatId);
			}
		}, this.getClass().getSimpleName() + ".Recalculation").start();
	}
    
    private void recalculateIfNecessary(final Integer flatId) {
    	synchronized (listeners) {
			for (ValueChangeListenerItem<IndoorClimateType> item : listeners) {
				if (item.getFlatId().equals(flatId)) {
					item.setValue(getIndoorClimateType(flatId, item.getPeriod(), item.getTimeUnit()));
				}
			}
		}
    }

    private void updateFlatCache() {
    	synchronized (flatCache) {
    		flatCache.clear();
    		flatCache.putAll(dataAccess.getFlatsAsMap());
			log.debug("Flats in cache: " + flatCache.size());
    	}
    }
}