package at.creadoo.homer.processing.data.impl;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;

import org.apache.commons.math3.stat.regression.SimpleRegression;
import org.apache.log4j.Logger;
import org.joda.time.DateTime;
import org.osgi.service.event.Event;
import org.osgi.service.event.EventHandler;

import at.ac.ait.hbs.homer.core.common.DataAccess;
import at.ac.ait.hbs.homer.core.common.enumerations.DeviceMessageType;
import at.ac.ait.hbs.homer.core.common.enumerations.DeviceType;
import at.ac.ait.hbs.homer.core.common.event.EventProperties;
import at.ac.ait.hbs.homer.core.common.event.EventTopic;
import at.ac.ait.hbs.homer.core.common.event.util.EventUtil;
import at.ac.ait.hbs.homer.core.common.model.DBDevice;
import at.ac.ait.hbs.homer.core.common.model.DBMessage;
import at.creadoo.homer.processing.data.BasicService;
import at.creadoo.homer.processing.data.Constants;
import at.creadoo.homer.processing.data.enumerations.AverageCalculationMode;
import at.creadoo.homer.processing.data.enumerations.CalculationMode;
import at.creadoo.homer.processing.data.listeners.ValueChangeListener;
import at.creadoo.homer.processing.data.util.Util;

public abstract class BasicServiceImpl implements BasicService<Double>, EventHandler {

	private static final Logger log = Logger.getLogger(BasicServiceImpl.class);

	private final DeviceType deviceType;

	private final DeviceMessageType deviceMessageType;
	
    private DataAccess dataAccess;
    
    private final Map<Integer, DBDevice> deviceCache = new HashMap<Integer, DBDevice>();
    
    private List<ValueChangeListenerItem<Double>> listeners = new ArrayList<ValueChangeListenerItem<Double>>();
	
    protected BasicServiceImpl(final DeviceType deviceType, final DeviceMessageType deviceMessageType) {
    	this.deviceType = deviceType;
    	this.deviceMessageType = deviceMessageType;
    }
    
    public void init() {
    	updateDeviceCache();
    }
    
    public abstract void destroy();
    
	public DataAccess getDataAccess() {
		return this.dataAccess;
	}
    
	public void setDataAccess(final DataAccess dataAccess) {
		this.dataAccess = dataAccess;
	}

    @Override
    public void handleEvent(final Event event) {
		try {
			if (EventUtil.isEventOfType(event, EventTopic.HOMECONTROL_DEVICES_CONFIGURATION_CHANGE_MESSAGE)) {
				log.debug("Reloading device cache");
				this.updateDeviceCache();
			} else if (EventUtil.isEventOfType(event, EventTopic.HOMECONTROL_DEVICE_MESSAGE)) {
				final DBMessage message = (DBMessage) event.getProperty(EventProperties.MESSAGE.name());
				if (containsDevice(message.getDeviceId()) && message.getMessageType().equals(deviceMessageType)) {
					new Thread(new Runnable() {
						@Override
						public void run() {
							recalculateIfNecessary(message);
						}
					}, this.getClass().getSimpleName() + ".Recalculation").start();
				}
			}
		} catch (Throwable ex) {
			log.error("Error while processing incoming event", ex);
		}
    }
    
    private void recalculateIfNecessary(final DBMessage message) {
    	if (message == null) {
    		return;
    	}
    	
    	if (!containsDevice(message.getDeviceId())) {
    		return;
    	}
    	
    	final DBDevice device = deviceCache.get(message.getDeviceId());
		if (device == null) {
			return;
		}
    	
    	synchronized (listeners) {
			for (ValueChangeListenerItem<Double> listener : listeners) {
				if (listener.hasDeviceId(message.getDeviceId())) {
					Double value = null;
					switch (listener.getCalculationMode()) {
					case AVERAGE:
						value = getValue(listener.getFlatId().intValue(), listener.getAverageCalculationMode(), listener.getPeriod(), listener.getTimeUnit(), listener.getExcludedDeviceIds());
						break;
					case ESTIMATE:
						value = estimateValue(listener.getFlatId().intValue(), listener.getPeriod(), listener.getTimeUnit(), listener.getExcludedDeviceIds());
						break;
					case LAST:
						value = getLastValue(listener.getFlatId().intValue(), listener.getAverageCalculationMode(), listener.getExcludedDeviceIds());
						break;
					}
					
					if (value == null || Double.isNaN(value)) {
						continue;
					}
					
					listener.setValue(value);
				}
			}
		}
    }
    
    @Override
	public final void addListenerForLastValue(final ValueChangeListener<Double> listener, final AverageCalculationMode mode, final int flatId) {
		synchronized (listeners) {
			if (!ListenerUtil.containsListener(listeners, listener)) {
				listeners.add(new ValueChangeListenerItem<Double>(listener, CalculationMode.LAST, mode, flatId, getDeviceIds(flatId), null));
			}
		}
	}

	@Override
	public final void removeListenerForLastValue(final ValueChangeListener<Double> listener) {
		ListenerUtil.removeListener(listeners, listener, CalculationMode.LAST);
	}
	
	@Override
	public final void addListenerForAverageValue(final ValueChangeListener<Double> listener, final AverageCalculationMode mode, final int flatId) {
		addListenerForAverageValue(listener, mode, flatId, Constants.DEFAULT_PERIOD, Constants.DEFAULT_UNIT, (List<Integer>) null);
	}
	
	@Override
	public final void addListenerForAverageValue(final ValueChangeListener<Double> listener, final AverageCalculationMode mode, final int flatId, final int... excludedDeviceIds) {
		addListenerForAverageValue(listener, mode, flatId, Constants.DEFAULT_PERIOD, Constants.DEFAULT_UNIT, excludedDeviceIds);
	}
	
	@Override
	public final void addListenerForAverageValue(final ValueChangeListener<Double> listener, final AverageCalculationMode mode, final int flatId, final List<Integer> excludedDevices) {
		addListenerForAverageValue(listener, mode, flatId, Constants.DEFAULT_PERIOD, Constants.DEFAULT_UNIT, excludedDevices);
	}
	
	@Override
	public final void addListenerForAverageValue(final ValueChangeListener<Double> listener, final AverageCalculationMode mode, final int flatId, final long calculationPeriod, final TimeUnit timeUnit) {
		addListenerForAverageValue(listener, mode, flatId, calculationPeriod, timeUnit, (List<Integer>) null);
	}
	
	@Override
	public final void addListenerForAverageValue(final ValueChangeListener<Double> listener, final AverageCalculationMode mode, final int flatId, final long calculationPeriod, final TimeUnit timeUnit, final int... excludedDeviceIds) {
		addListenerForAverageValue(listener, mode, flatId, calculationPeriod, timeUnit, getDeviceIds(excludedDeviceIds));
	}
	
	@Override
	public final void addListenerForAverageValue(final ValueChangeListener<Double> listener, final AverageCalculationMode mode, final int flatId, final long calculationPeriod, final TimeUnit timeUnit, final List<Integer> excludedDeviceIds) {
		synchronized (listeners) {
			if (!ListenerUtil.containsListener(listeners, listener)) {
				listeners.add(new ValueChangeListenerItem<Double>(listener, CalculationMode.AVERAGE, mode, flatId, getDeviceIds(flatId, excludedDeviceIds), excludedDeviceIds, calculationPeriod, timeUnit));
			}
		}
	}

	@Override
	public final void removeListenerForAverageValue(final ValueChangeListener<Double> listener) {
		ListenerUtil.removeListener(listeners, listener, CalculationMode.AVERAGE);
	}
	
	@Override
	public final void addListenerForEstimateValue(final ValueChangeListener<Double> listener, final int flatId) {
		addListenerForEstimateValue(listener, flatId, Constants.DEFAULT_PERIOD, Constants.DEFAULT_UNIT, (List<Integer>) null);
	}
	
	@Override
	public final void addListenerForEstimateValue(final ValueChangeListener<Double> listener, final int flatId, final int... excludedDeviceIds) {
		addListenerForEstimateValue(listener, flatId, Constants.DEFAULT_PERIOD, Constants.DEFAULT_UNIT, excludedDeviceIds);
	}
	
	@Override
	public final void addListenerForEstimateValue(final ValueChangeListener<Double> listener, final int flatId, final List<Integer> excludedDevices) {
		addListenerForEstimateValue(listener, flatId, Constants.DEFAULT_PERIOD, Constants.DEFAULT_UNIT, excludedDevices);
	}
	
	@Override
	public final void addListenerForEstimateValue(final ValueChangeListener<Double> listener, final int flatId, final long calculationPeriod, final TimeUnit timeUnit) {
		addListenerForEstimateValue(listener, flatId, calculationPeriod, timeUnit, (List<Integer>) null);
	}
	
	@Override
	public final void addListenerForEstimateValue(final ValueChangeListener<Double> listener, final int flatId, final long calculationPeriod, final TimeUnit timeUnit, final int... excludedDeviceIds) {
		addListenerForEstimateValue(listener, flatId, calculationPeriod, timeUnit, getDeviceIds(excludedDeviceIds));
	}
	
	@Override
	public final void addListenerForEstimateValue(final ValueChangeListener<Double> listener, final int flatId, final long calculationPeriod, final TimeUnit timeUnit, final List<Integer> excludedDeviceIds) {
		synchronized (listeners) {
			if (!ListenerUtil.containsListener(listeners, listener)) {
				listeners.add(new ValueChangeListenerItem<Double>(listener, CalculationMode.ESTIMATE, null, flatId, getDeviceIds(flatId, excludedDeviceIds), excludedDeviceIds, calculationPeriod, timeUnit));
			}
		}
	}

	@Override
	public final void removeListenerForEstimateValue(final ValueChangeListener<Double> listener) {
		ListenerUtil.removeListener(listeners, listener, CalculationMode.ESTIMATE);
	}

	@Override
	public final void removeListener(final ValueChangeListener<Double> listener) {
		ListenerUtil.removeListener(listeners, listener);
	}


	@Override
	public final List<Integer> getDeviceIds(final int flatId) {
		return Util.getDeviceIds(dataAccess, deviceType, flatId);
	}

	@Override
	public final List<Integer> getDeviceIds(final int flatId, final List<Integer> excludedDeviceIds) {
		return Util.getDeviceIds(dataAccess, deviceType, flatId, excludedDeviceIds);
	}

	@Override
	public final List<Integer> getDeviceIds(final int flatId, final int... excludedDeviceIds) {
		return Util.getDeviceIds(dataAccess, deviceType, flatId, excludedDeviceIds);
	}

	@Override
	public final List<Integer> getDeviceIds(final Collection<DBDevice> devices) {
		return Util.getDeviceIds(devices);
	}

	@Override
	public final List<Integer> getDeviceIds(final List<DBDevice> devices) {
		return Util.getDeviceIds(devices);
	}

	@Override
	public final List<Integer> getDeviceIds(final int... devices) {
		return Util.getDeviceIds(devices);
	}
	

	
	@Override
	public final List<DBMessage> getDeviceMessages(final int flatId) {
		return getDeviceMessages(flatId, Constants.DEFAULT_PERIOD, Constants.DEFAULT_UNIT, (List<Integer>) null);
	}
	
	@Override
	public final List<DBMessage> getDeviceMessages(final int flatId, final int... excludedDeviceIds) {
		return getDeviceMessages(flatId, Constants.DEFAULT_PERIOD, Constants.DEFAULT_UNIT, excludedDeviceIds);
	}
	
	@Override
	public final List<DBMessage> getDeviceMessages(final int flatId, final List<Integer> excludedDevices) {
		return getDeviceMessages(flatId, Constants.DEFAULT_PERIOD, Constants.DEFAULT_UNIT, excludedDevices);
	}
	
	@Override
	public final List<DBMessage> getDeviceMessages(final int flatId, final long calculationPeriod, final TimeUnit timeUnit) {
		return getDeviceMessages(flatId, calculationPeriod, timeUnit, (List<Integer>) null);
	}
	
	@Override
	public final List<DBMessage> getDeviceMessages(final int flatId, final long calculationPeriod, final TimeUnit timeUnit, final int... excludedDeviceIds) {
		return getDeviceMessages(flatId, calculationPeriod, timeUnit, getDeviceIds(excludedDeviceIds));
	}
	
	@Override
	public List<DBMessage> getDeviceMessages(final int flatId, final long calculationPeriod, final TimeUnit timeUnit, final List<Integer> excludedDeviceIds) {
		final List<DBMessage> result = new ArrayList<DBMessage>();
		
		final List<Integer> deviceIds = getDeviceIds(flatId, excludedDeviceIds);
		if (deviceIds == null || deviceIds.isEmpty()) {
			return result;
		}
		
		for (DBMessage message : getMessagesForPeriod(calculationPeriod, timeUnit, deviceIds)) {
			if (message == null) {
				continue;
			}
			
			if (message.getData1() != -1) {
				result.add(message);
			}
		}
		
		return result;
	}
	
	@Override
	public Double getLastValue(final int flatId) {
		return getLastValue(flatId, Constants.DEFAULT_MODE, (List<Integer>) null);
	}
		
	@Override
	public Double getLastValue(final int flatId, final int... excludedDeviceIds) {
		return getLastValue(flatId, Constants.DEFAULT_MODE, excludedDeviceIds);
	}
	
	@Override
	public Double getLastValue(final int flatId, final List<Integer> excludedDevices) {
		return getLastValue(flatId, Constants.DEFAULT_MODE, excludedDevices);
	}
	
	@Override
	public Double getLastValue(final int flatId, final AverageCalculationMode mode) {
		return getLastValue(flatId, mode, (List<Integer>) null);
	}
		
	@Override
	public Double getLastValue(final int flatId, final AverageCalculationMode mode, final int... excludedDeviceIds) {
		return getLastValue(flatId, mode, getDeviceIds(excludedDeviceIds));
	}
	
	@Override
	public Double getLastValue(final int flatId, final AverageCalculationMode mode, final List<Integer> excludedDevices) {
		if (mode == null) {
			return Double.NaN;
		}
		
		if (mode != AverageCalculationMode.MEAN && mode != AverageCalculationMode.MEDIAN) {
			return Double.NaN;
		}
		
		final List<Integer> deviceIds = getDeviceIds(flatId, excludedDevices);
		if (deviceIds == null || deviceIds.isEmpty()) {
			return Double.NaN;
		}
		
		final Map<Integer, Double> deviceValues = new HashMap<Integer, Double>();

		for (Integer deviceId : deviceIds) {
			final DBMessage message = dataAccess.getMessageLatest(deviceId, deviceMessageType);
			if (message == null) {
				continue;
			}
			
			if (message.getData1() != -1) {
				deviceValues.put(deviceId, message.getData1());
			}
		}
		
		return Util.average(deviceValues.values(), mode); 
	}
	
	@Override
	public Double getValue(final int flatId) {
		return getValue(flatId, Constants.DEFAULT_MODE, (List<Integer>) null);
	}
		
	@Override
	public Double getValue(final int flatId, final int... excludedDeviceIds) {
		return getValue(flatId, Constants.DEFAULT_MODE, excludedDeviceIds);
	}
	
	@Override
	public Double getValue(final int flatId, final List<Integer> excludedDevices) {
		return getValue(flatId, Constants.DEFAULT_MODE, excludedDevices);
	}

	@Override
	public Double getValue(int flatId, long calculationPeriod, TimeUnit timeUnit) {
		return getValue(flatId, Constants.DEFAULT_MODE, calculationPeriod, timeUnit);
	}

	@Override
	public Double getValue(int flatId, long calculationPeriod, TimeUnit timeUnit, int... excludedDeviceIds) {
		return getValue(flatId, Constants.DEFAULT_MODE, calculationPeriod, timeUnit, excludedDeviceIds);
	}

	@Override
	public Double getValue(int flatId, long calculationPeriod, TimeUnit timeUnit, List<Integer> excludedDevices) {
		return getValue(flatId, Constants.DEFAULT_MODE, calculationPeriod, timeUnit, excludedDevices);
	}

	@Override
	public final Double getValue(final int flatId, final AverageCalculationMode mode) {
		return getValue(flatId, mode, Constants.DEFAULT_PERIOD, Constants.DEFAULT_UNIT, (List<Integer>) null);
	}

	@Override
	public final Double getValue(final int flatId, final AverageCalculationMode mode, final List<Integer> excludedDevices) {
		return getValue(flatId, mode, Constants.DEFAULT_PERIOD, Constants.DEFAULT_UNIT, excludedDevices);
	}

	@Override
	public final Double getValue(final int flatId, final AverageCalculationMode mode, final int... excludedDeviceIds) {
		return getValue(flatId, mode, Constants.DEFAULT_PERIOD, Constants.DEFAULT_UNIT, excludedDeviceIds);
	}

	@Override
	public final Double getValue(final int flatId, final AverageCalculationMode mode, final long calculationPeriod, final TimeUnit timeUnit) {
		return getValue(flatId, mode, calculationPeriod, timeUnit, (List<Integer>) null);
	}

	@Override
	public final Double getValue(final int flatId, final AverageCalculationMode mode, final long calculationPeriod, final TimeUnit timeUnit, final int... excludedDeviceIds) {
		return getValue(flatId, mode, calculationPeriod, timeUnit, getDeviceIds(excludedDeviceIds));
	}

	@Override
	public Double getValue(final int flatId, final AverageCalculationMode mode, final long calculationPeriod, final TimeUnit timeUnit, final List<Integer> excludedDevices) {
		if (calculationPeriod == 0 || timeUnit == null) {
			return null;
		}

		final List<Double> deviceValues = new ArrayList<Double>();
		for (DBMessage message : getDeviceMessages(flatId, calculationPeriod, timeUnit, excludedDevices)) {
			deviceValues.add(message.getData1());
		}
		
		return Util.average(deviceValues, mode);
	}

	@Override
	public final Double estimateValue(final int flatId) {
		return estimateValue(flatId, Constants.DEFAULT_PERIOD, Constants.DEFAULT_UNIT, (List<Integer>) null);
	}

	@Override
	public final Double estimateValue(final int flatId, final List<Integer> excludedDevices) {
		return estimateValue(flatId, Constants.DEFAULT_PERIOD, Constants.DEFAULT_UNIT, excludedDevices);
	}

	@Override
	public final Double estimateValue(final int flatId, final int... excludedDeviceIds) {
		return estimateValue(flatId, Constants.DEFAULT_PERIOD, Constants.DEFAULT_UNIT, excludedDeviceIds);
	}

	@Override
	public final Double estimateValue(final int flatId, final long calculationPeriod, final TimeUnit timeUnit) {
		return estimateValue(flatId, calculationPeriod, timeUnit, (List<Integer>) null);
	}

	@Override
	public final Double estimateValue(final int flatId, final long calculationPeriod, final TimeUnit timeUnit, final int... excludedDeviceIds) {
		return estimateValue(flatId, calculationPeriod, timeUnit, getDeviceIds(excludedDeviceIds));
	}
	
	@Override
	public final Double estimateValue(final int flatId, final long calculationPeriod, final TimeUnit timeUnit, final List<Integer> excludedDevices) {
		return estimateValue(flatId, DateTime.now(), calculationPeriod, timeUnit, excludedDevices);
	}

	@Override
	public final Double estimateValue(final int flatId, final DateTime forTimestamp) {
		return estimateValue(flatId, forTimestamp, Constants.DEFAULT_PERIOD, Constants.DEFAULT_UNIT, (List<Integer>) null);
	}

	@Override
	public final Double estimateValue(final int flatId, final DateTime forTimestamp, final List<Integer> excludedDevices) {
		return estimateValue(flatId, forTimestamp, Constants.DEFAULT_PERIOD, Constants.DEFAULT_UNIT, excludedDevices);
	}

	@Override
	public final Double estimateValue(final int flatId, final DateTime forTimestamp, final int... excludedDeviceIds) {
		return estimateValue(flatId, forTimestamp, Constants.DEFAULT_PERIOD, Constants.DEFAULT_UNIT, excludedDeviceIds);
	}

	@Override
	public final Double estimateValue(final int flatId, final DateTime forTimestamp, final long calculationPeriod, final TimeUnit timeUnit) {
		return estimateValue(flatId, forTimestamp, calculationPeriod, timeUnit, (List<Integer>) null);
	}

	@Override
	public final Double estimateValue(final int flatId, final DateTime forTimestamp, final long calculationPeriod, final TimeUnit timeUnit, final int... excludedDeviceIds) {
		return estimateValue(flatId, forTimestamp, calculationPeriod, timeUnit, getDeviceIds(excludedDeviceIds));
	}

	@Override
	public Double estimateValue(final int flatId, final DateTime forTimestamp, final long calculationPeriod, final TimeUnit timeUnit, final List<Integer> excludedDevices) {
		if (forTimestamp == null || calculationPeriod == 0 || timeUnit == null) {
			return null;
		}
		
		final SimpleRegression simpleRegression = new SimpleRegression(true);
		for (DBMessage message : getDeviceMessages(flatId, calculationPeriod, timeUnit, excludedDevices)) {
			simpleRegression.addData(message.getTimestamp().getMillis(), message.getData1());
		}
		
		return Util.round(simpleRegression.predict(DateTime.now().getMillis()), 2);
	}
	
	protected final List<DBMessage> getMessagesForPeriod(final long duration, final TimeUnit timeUnit, final Integer deviceId) {
		final DateTime timestamp = Util.getTimestampForPeriod(duration, timeUnit);
		if (timestamp == null) {
			return new ArrayList<DBMessage>();
		}
		return dataAccess.getMessagesAfterByDeviceIdAndMessageType(timestamp.getMillis(), deviceId, deviceMessageType);
	}
	
	protected final List<DBMessage> getMessagesForPeriod(final long duration, final TimeUnit timeUnit, final List<Integer> deviceIds) {
		final DateTime timestamp = Util.getTimestampForPeriod(duration, timeUnit);
		if (timestamp == null) {
			return new ArrayList<DBMessage>();
		}
		return dataAccess.getMessagesAfterByDeviceIdsAndMessageType(timestamp.getMillis(), deviceIds, deviceMessageType);
	}
	
	private final boolean containsDevice(final Integer deviceId) {
		if (deviceId == null) {
			return false;
		}
		
		synchronized (deviceCache) {
			if (deviceCache.containsKey(deviceId)) {
				return true;
			}
		}
		return false;
	}

    private void updateDeviceCache() {
    	synchronized (deviceCache) {
	    	deviceCache.clear();
	    	deviceCache.putAll(dataAccess.getDevicesByDeviceTypeAsMap(deviceType));
			log.debug("Devices in cache: " + deviceCache.size());
    	}
    }

}