/*
 * Copyright 2016 crea-doo.at
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License. 
 */
package at.creadoo.homer.hwabstraction.toughswitch;

import at.ac.ait.hbs.homer.core.common.model.DBMessage;
import at.ac.ait.hbs.homer.core.common.device.DeviceValue;
import at.ac.ait.hbs.homer.core.common.enumerations.BatteryStatus;
import at.ac.ait.hbs.homer.core.common.enumerations.DeviceMessageType;
import at.ac.ait.hbs.homer.core.common.enumerations.DeviceProtocol;
import at.ac.ait.hbs.homer.core.common.enumerations.DeviceType;
import at.ac.ait.hbs.homer.core.common.event.EventBuilder;
import at.ac.ait.hbs.homer.core.common.event.EventProperties;
import at.ac.ait.hbs.homer.core.common.event.EventTopic;
import at.ac.ait.hbs.homer.core.common.event.util.EventUtil;
import at.ac.ait.hbs.homer.core.common.exception.DeviceMessageTypeNotUniqueException;
import at.ac.ait.hbs.homer.core.common.scenario.helper.Device;
import at.creadoo.homer.hwabstraction.toughswitch.util.Util;
import at.creadoo.util.toughswitch.PortPOEStatus;
import at.creadoo.util.toughswitch.ToughSwitch;
import at.creadoo.util.toughswitch.ToughSwitchBuilder;
import at.creadoo.util.toughswitch.ToughSwitchException;

import org.apache.log4j.Logger;
import org.osgi.service.cm.ConfigurationAdmin;
import org.osgi.service.cm.ConfigurationException;
import org.osgi.service.cm.ManagedService;
import org.osgi.service.event.Event;
import org.osgi.service.event.EventAdmin;
import org.osgi.service.event.EventHandler;

import java.io.IOException;
import java.util.Dictionary;
import java.util.HashMap;
import java.util.Map;
import java.util.TreeMap;

/**
 * <p>
 * Title: ToughSwitchHardware
 * </p>
 * <p/>
 *
 * @author crea-doo
 * @version 1.0.0
 */

public class ToughSwitchHardware implements ManagedService, EventHandler {

    private static final Logger log = Logger.getLogger(ToughSwitchHardware.class);
    
    private ToughSwitch toughSwitch;
    private String toughSwitchMAC;
    private Long toughSwitchId;
	private Map<Long, Integer> toughSwitchPortIds = new HashMap<Long, Integer>();
	private Map<Integer, PortPOEStatus> toughSwitchPortDefaults = new HashMap<Integer, PortPOEStatus>();

    private ConfigurationAdmin configurationAdmin;

    private EventAdmin eventAdmin;
    
    public ToughSwitchHardware() {
		log.info("Initializing " + this.getClass().getSimpleName());
    }
    
    public void init() {
		log.info("Starting " + this.getClass().getSimpleName());

		final Map<String, String> props = Util.toMap(getConfiguration());
		
		updatePortDefaults(props);
		
		final String ipHost = Util.getPropertyValue(props, Constants.KEY_IP_HOST);
		final String ipPort = Util.getPropertyValue(props, Constants.KEY_IP_PORT);
		final String username = Util.getPropertyValue(props, Constants.KEY_USERNAME);
		final String password = Util.getPropertyValue(props, Constants.KEY_PASSWORD);
		
		int ipPortInt = ToughSwitchBuilder.DEFAULT_PORT;
		try {
			ipPortInt = Integer.parseInt(ipPort);
		} catch (NumberFormatException ex) {
			//
		}

		toughSwitch = new ToughSwitchBuilder(ipHost, ipPortInt).setUsername(username).setPassword(password).build();
		try {
			if (toughSwitch == null || !toughSwitch.connect()) {
				log.error("Error while initializing connection to ToughSwitch");
				return;
			}
		} catch (ToughSwitchException ex) {
			log.error("Error while initializing connection to ToughSwitch", ex);
			return;
		}
		
		try {
			this.toughSwitchMAC = toughSwitch.getMAC();
			if (toughSwitchMAC == null) {
				log.error("Error while reading MAC address from ToughSwitch");
				return;
			}
			
			log.info("Connected to ToughSwitch device '" + toughSwitchMAC + "' (" + ipHost + ":" + ipPort + ")");

			this.toughSwitchId = getDeviceId(this.toughSwitchMAC);
			log.info("HardwareID for device is '" + this.toughSwitchId + "'");
			
			toughSwitchPortIds.put(getPortId(this.toughSwitchId, 1), 1);
			toughSwitchPortIds.put(getPortId(this.toughSwitchId, 2), 2);
			toughSwitchPortIds.put(getPortId(this.toughSwitchId, 3), 3);
			toughSwitchPortIds.put(getPortId(this.toughSwitchId, 4), 4);
			toughSwitchPortIds.put(getPortId(this.toughSwitchId, 5), 5);
			toughSwitchPortIds.put(getPortId(this.toughSwitchId, 6), 6);
			toughSwitchPortIds.put(getPortId(this.toughSwitchId, 7), 7);
			toughSwitchPortIds.put(getPortId(this.toughSwitchId, 8), 8);
			
			for (final Long port : new TreeMap<Long, Integer>(toughSwitchPortIds).keySet()) {
				log.info("HardwareID for Port " + toughSwitchPortIds.get(port) + " is '" + port + "'");
			}
		} catch (ToughSwitchException ex) {
			log.error("Error while reading MAC address", ex);
		}
    }
    
	/**
	 * Destroy method called by container. Shut down gracefully
	 */
	public void destroy() {
		log.info("Stopping " + this.getClass().getSimpleName());
		
		if (toughSwitch != null) {
			toughSwitch.disconnect();
		}
		
		toughSwitch = null;
	}
    
    private Dictionary<String, ?> getConfiguration() {
    	try {
			return configurationAdmin.getConfiguration(Constants.CONFIG_PID, "?").getProperties();
		} catch (IOException ex) {
			log.error("Error loading configuration", ex);
		}
    	return null;
    }

    @Override
    public void handleEvent(final Event event) {
		log.debug("Received " + event);
		try {
			if (EventUtil.isEventOfType(event, EventTopic.HOMECONTROL_ACTUATOR_REQUEST)) {
				final Device device = (Device) event.getProperty(EventProperties.DEVICE.name());
				final DeviceValue deviceValue = (DeviceValue) event.getProperty(EventProperties.DEVICE_VALUE.name());

				log.debug("Actuator request: " + device);

				if (!DeviceProtocol.UNSPECIFIED.equals(device.getProtocol())) {
					log.debug("Device protocol '" + device.getProtocol() + "' not supported!");
					return;
				}
				if (device.getHardwareId() == null) {
					log.debug("Actuator with id " + device.getDeviceId() + " has no hardware id!");
					return;
				}
				if (!DeviceType.MDC_AI_TYPE_ACTUATOR_SWITCH.equals(device.getType())) {
					log.debug("Device type '" + device.getType().name() + "' not supported!");
					return;
				}
				
				final Boolean on = DeviceMessageType.MDC_AI_FLAG_SENSOR_SWITCH_ON.getValue().equals(((Integer) deviceValue.getValue()));
				
				if (device.getHardwareId().equals(this.toughSwitchId)) {
					if (device.getHardwareId2() != null && toughSwitch.isPortValid(device.getHardwareId2().intValue())) {
						// Switch the port identified by device id and port
						final int port = device.getHardwareId2().intValue();
						if (on) {
							log.debug("Switch port " + port + " on");
							final PortPOEStatus status = toughSwitchPortDefaults.get(port);
							if (status != null) {
								this.toughSwitch.setPortPOE(port, status);
							}
						} else {
							log.debug("Switch port " + port + " off");
							this.toughSwitch.setPortPOE(port, PortPOEStatus.OFF);
						}
					} else {
						// Switch all ports
						if (on) {
							log.debug("Switch all ports on");
							for (int port = 1; port <= ToughSwitch.MAX_PORTS; port++) {
								final PortPOEStatus status = toughSwitchPortDefaults.get(port);
								if (status != null) {
									this.toughSwitch.setPortPOE(port, status);
								}
							}
						} else {
							log.debug("Switch all ports off");
							this.toughSwitch.setPortsPOE(PortPOEStatus.OFF);
						}
					}
					sendActuatorMessage(device, deviceValue);
				} else if (toughSwitchPortIds.containsKey(device.getHardwareId())) {
					// Switch the port identified by port id
					final Integer port = toughSwitchPortIds.get(device.getHardwareId());
					if (on) {
						log.debug("Switch port " + port + " on");
						final PortPOEStatus status = toughSwitchPortDefaults.get(port);
						if (status != null) {
							this.toughSwitch.setPortPOE(port, status);
						}
					} else {
						log.debug("Switch port " + port + " off");
						this.toughSwitch.setPortPOE(port, PortPOEStatus.OFF);
					}
					sendActuatorMessage(device, deviceValue);
				} else {
					//
				}
			}
		} catch (Throwable ex) {
			log.error("Error while processing incoming event", ex);
		}
    }

	private void sendActuatorMessage(final Device device, final DeviceValue deviceValue) {
		sendActuatorMessage(device, deviceValue, null, null);
    }

	private void sendActuatorMessage(final Device device, final DeviceValue deviceValue, final Double data1, final Double data2) {
        log.debug("Sending actuator message");

        final DBMessage message = new DBMessage();
        message.setDeviceId(device.getDeviceId());
        try {
			message.setMessageType(DeviceMessageType.getFlag(device.getType(), deviceValue));
		} catch (DeviceMessageTypeNotUniqueException ex) {
			log.error("Error setting DeviceMessageType", ex);
		}
        message.setTimestamp(System.currentTimeMillis());
        message.setBatteryStatus(BatteryStatus.NO_CONDITION_DETECTED);

		if (data1 != null) {
			message.setData1(data1);
		}
		if (data2 != null) {
			message.setData2(data2);
		}
        
		final Event event = EventBuilder.createActuatorMessageEvent(message, deviceValue);
		
        eventAdmin.postEvent(event);
    }

    private Long getDeviceId(final String deviceMAC) {
    	if (Util.isValidMacAddress(deviceMAC)) {
    		final String tempDeviceMAC = deviceMAC.replaceAll("[^a-fA-F0-9]", "");
    		final String hexID = tempDeviceMAC.substring(tempDeviceMAC.length() - 6);
    		
	    	try {
	    		return Long.parseLong(hexID, 16);
	    	} catch (NumberFormatException ex) {
	    		//
	    	}
	    }
    	return null;
    }

    private Long getPortId(final Long deviceId, final Integer port) {
    	if (toughSwitch.isPortValid(port)) {
    		return (deviceId * 10) + port;
    	}
    	return null;
    }

	private void updatePortDefaults(final Dictionary<String, ?> properties) {
		updatePortDefaults(Util.toMap(properties));
	}
	
	private void updatePortDefaults(final Map<String, String> properties) {
		for (int port = 1; port <= ToughSwitch.MAX_PORTS; port++) {
			final String defaultValue = Util.getPropertyValue(properties, Constants.KEY_PORT_PREFIX + port);
			if (defaultValue != null) {
				final PortPOEStatus portStatus = PortPOEStatus.getPortStatus(defaultValue);
				if (portStatus != null) {
					toughSwitchPortDefaults.put(port, portStatus);
				}
			}
		}
	}

	/**
	 * Setter of {@link ConfigurationAdmin} for blueprint
	 * 
	 * @param configurationAdmin The {@link ConfigurationAdmin} instance
	 */
    public final void setConfigurationAdmin(final ConfigurationAdmin configurationAdmin) {
    	this.configurationAdmin = configurationAdmin;
    }

	/**
	 * Setter of {@link EventAdmin} for blueprint
	 * 
	 * @param eventAdmin The {@link EventAdmin} instance
	 */
	public final void setEventAdmin(final EventAdmin eventAdmin) {
		this.eventAdmin = eventAdmin;
	}

	@Override
	public void updated(final Dictionary<String, ?> properties) throws ConfigurationException {
		if (properties != null) {
			updatePortDefaults(properties);
		}
	}

}
