/*
 * Copyright 2016 crea-doo.at
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License. 
 */
package at.creadoo.homer.hwabstraction.toughswitch.util;

import java.util.Dictionary;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.regex.Pattern;

/**
 * Provides small utils for the hardware bundle.
 */
public final class Util {

	private Util() {
		//
	}
	
	public static String dataLog(byte data) {
		return dataLog(new byte[] { data });
	}
	
	public static String dataLog(byte[] data) {
		final StringBuilder sb = new StringBuilder();
		for (int i = 0; i < data.length; i++) {
			sb.append(String.format("%02x ", Byte.valueOf(data[i])).toUpperCase());
		}
		return sb.toString();
	}
    
	public static String toHex(final byte data) {
		return String.format("%02x", Byte.valueOf(data)).toUpperCase();
	}

	public static String toHex(final long data) {
		return String.format("%02x", Long.valueOf(data)).toUpperCase();
	}
    
	public static byte[] toByteFromHex(final String s) {
		int len = s.length();
		byte[] data = new byte[len / 2];
		for (int i = 0; i < len; i += 2) {
			data[i / 2] = (byte) ((Character.digit(s.charAt(i), 16) << 4) + Character.digit(s.charAt(i + 1), 16));
		}
		return data;
	}
	
    public static String hardwareIdsToString(final List<Long> hardwareIds) {
		final StringBuffer strbuf = new StringBuffer();
		int index = 1;
		for (final Long hardwareId: hardwareIds) {
			strbuf.append(hardwareIdToString(hardwareId, index));
			if (index < hardwareIds.size()) {
				strbuf.append("; ");
			} else {
				
			}
			index++;
		}
		return strbuf.toString();
	}
	
    public static String hardwareIdToString(final Long hardwareId) {
		return hardwareIdToString(hardwareId, null);
	}
	
    public static String hardwareIdToString(final Long hardwareId, final Integer index) {
		if (index != null) {
			return "HardwareId " + index +": " + hardwareId +" (" + toHex(hardwareId) + ")";
		} else {
			return "HardwareId: " + hardwareId +" (" + toHex(hardwareId) + ")";
		}
	}

    public static boolean isValidMacAddress(final String mac) {
		final Pattern patternMacPairs = Pattern.compile("^([a-fA-F0-9]{2}[:\\.-]?){5}[a-fA-F0-9]{2}$");
		final Pattern patternMacTriples = Pattern.compile("^([a-fA-F0-9]{3}[:\\.-]?){3}[a-fA-F0-9]{3}$");

		// Mac addresses usually are 6 * 2 hex nibbles separated by colons,
		// but apparently it is legal to have 4 * 3 hex nibbles as well,
		// and the separators can be any of : or - or . or nothing.
		return (patternMacPairs.matcher(mac).find() || patternMacTriples.matcher(mac).find());
	}

	public static Map<String, String> toMap(final Dictionary<String, ?> dictionary) {
		if (dictionary == null) {
			return new HashMap<String, String>();
		}
		final Map<String, String> map = new HashMap<String, String>(dictionary.size());
		Enumeration<String> keys = dictionary.keys();
		while (keys.hasMoreElements()) {
			String key = keys.nextElement();
			map.put(key, dictionary.get(key).toString());
		}
		return map;
	}
	
	public static String getPropertyValue(final Dictionary<String, ?> properties, final String key) {
		return getPropertyValue(Util.toMap(properties), key);
	}
	
	public static String getPropertyValue(Map<String, String> properties, final String key) {
		if (properties != null && properties.containsKey(key)) {
			return properties.get(key);
		}
		return null;
	}
}
