/*
 * Copyright 2016 crea-doo.at
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License. 
 */
package at.creadoo.homer.hwabstraction.netio;

import at.ac.ait.hbs.homer.core.common.model.DBDevice;
import at.ac.ait.hbs.homer.core.common.model.DBMessage;
import at.ac.ait.hbs.homer.core.common.enumerations.BatteryStatus;
import at.ac.ait.hbs.homer.core.common.enumerations.DeviceCategoryType;
import at.ac.ait.hbs.homer.core.common.enumerations.DeviceMessageType;
import at.ac.ait.hbs.homer.core.common.enumerations.DeviceProtocol;
import at.ac.ait.hbs.homer.core.common.enumerations.DeviceType;
import at.ac.ait.hbs.homer.core.common.enumerations.MetaKey;
import at.ac.ait.hbs.homer.core.common.event.EventBuilder;
import at.ac.ait.hbs.homer.core.common.event.EventProperties;
import at.ac.ait.hbs.homer.core.common.event.EventTopic;
import at.ac.ait.hbs.homer.core.common.event.util.EventUtil;
import at.ac.ait.hbs.homer.core.common.util.PropertyUtil;
import at.creadoo.util.netio.NetIO;
import at.creadoo.util.netio.NetIOBuilder;
import at.creadoo.util.netio.NetIOException;

import org.apache.log4j.Logger;
import org.osgi.service.event.Event;
import org.osgi.service.event.EventAdmin;
import org.osgi.service.event.EventHandler;

import java.util.HashMap;
import java.util.Map;
import java.util.TreeMap;
import java.util.regex.Pattern;

/**
 * <p>
 * Title: NetIOHardware
 * </p>
 * <p/>
 *
 * @author crea-doo
 * @version 1.0.0
 */

public class NetIOHardware implements EventHandler {

    private static final Logger log = Logger.getLogger(NetIOHardware.class);
    
    private NetIO netIO;
    private String netioMAC;
    private Long netioId;
	private Map<Long, Integer> netioPortIds = new HashMap<Long, Integer>();
    
	private String ipHost;
	private Integer ipPort;
	private String username;
	private String password;

    private EventAdmin eventAdmin;
    
    public NetIOHardware() {
		log.info("Initializing " + this.getClass().getSimpleName());
    }

    public void init() {
		log.info("Starting " + this.getClass().getSimpleName());

		netIO = new NetIOBuilder(ipHost, ipPort).setUsername(username).setPassword(password).build();
		
		if (netIO == null) {
			log.error("Error while initializing connection to NetIO");
			return;
		}
		
		try {
			this.netioMAC = netIO.getMAC();
			log.info("Connected to NetIO device '" + netioMAC + "' (" + ipHost + ":" + ipPort + ")");

			this.netioId = getDeviceId(this.netioMAC);
			log.info("HardwareID for device is '" + this.netioId + "'");
			
			netioPortIds.put(getPortId(this.netioId, 1), 1);
			netioPortIds.put(getPortId(this.netioId, 2), 2);
			netioPortIds.put(getPortId(this.netioId, 3), 3);
			netioPortIds.put(getPortId(this.netioId, 4), 4);
			
			for (final Long port : new TreeMap<Long, Integer>(netioPortIds).keySet()) {
				log.info("HardwareID for Port " + netioPortIds.get(port) + " is '" + port + "'");
			}
		} catch (NetIOException ex) {
			log.error("Error while reading MAC address", ex);
		}
    }
    
	/**
	 * Destroy method called by container. Shut down gracefully
	 */
	public void destroy() {
		log.info("Stopping " + this.getClass().getSimpleName());
		
		netIO = null;
	}

    @Override
    public void handleEvent(final Event event) {
		try {
			if (EventUtil.isEventOfType(event, EventTopic.HOMECONTROL_DEVICE_REQUEST)) {
				final DBDevice device = (DBDevice) event.getProperty(EventProperties.DEVICE.name());
				final DeviceMessageType deviceMessageType = (DeviceMessageType) event.getProperty(EventProperties.DEVICE_MESSAGE_TYPE.name());
				//final DeviceValue deviceValue = (DeviceValue) event.getProperty(EventProperties.DEVICE_VALUE.name());

				if (device.getType().getCategoryType().equals(DeviceCategoryType.MDC_AI_TYPE_ACTUATOR)) {
					log.debug("Actuator request: " + device);

					if (!DeviceProtocol.UNSPECIFIED.equals(device.getProtocol())) {
						log.debug("Device protocol '" + device.getProtocol() + "' not supported!");
						return;
					}
					if (!device.hasMetaValue(MetaKey.DEVICE_HARDWAREID)) {
						log.debug("Actuator with id " + device.getId() + " has no hardware id!");
						return;
					}
					if (!DeviceType.MDC_AI_TYPE_ACTUATOR_SWITCH.equals(device.getType())) {
						log.debug("Device type '" + device.getType().name() + "' not supported!");
						return;
					}

					final Boolean on = DeviceMessageType.MDC_AI_FLAG_SENSOR_SWITCH_ON.equals(deviceMessageType);
					
					if (device.getLongMetaValue(MetaKey.DEVICE_HARDWAREID).equals(this.netioId)) {
						if (device.hasMetaValue(MetaKey.DEVICE_HARDWAREID2) && netIO.isPortValid(device.getLongMetaValue(MetaKey.DEVICE_HARDWAREID2).intValue())) {
							final int hardwareId2 = device.getLongMetaValue(MetaKey.DEVICE_HARDWAREID2).intValue();
							// Switch the port identified by device id and port
							if (on) {
								log.debug("Switch port " + hardwareId2 + " on");
								this.netIO.setPortOn(hardwareId2);
							} else {
								log.debug("Switch port " + hardwareId2 + " off");
								this.netIO.setPortOff(hardwareId2);
							}
						} else {
							// Switch all ports
							if (on) {
								log.debug("Switch all ports on");
								this.netIO.setPortsOn();
							} else {
								log.debug("Switch all ports off");
								this.netIO.setPortsOff();
							}
						}
						sendDeviceMessage(device, deviceMessageType);
					} else if (netioPortIds.containsKey(device.getLongMetaValue(MetaKey.DEVICE_HARDWAREID))) {
						final Long hardwareId = device.getLongMetaValue(MetaKey.DEVICE_HARDWAREID);
						// Switch the port identified by port id
						if (on) {
							log.debug("Switch port " + netioPortIds.get(hardwareId) + " on");
							this.netIO.setPortOn(netioPortIds.get(hardwareId));
						} else {
							log.debug("Switch port " + netioPortIds.get(hardwareId) + " off");
							this.netIO.setPortOff(netioPortIds.get(hardwareId));
						}
						sendDeviceMessage(device, deviceMessageType);
					} else {
						//
					}
				}
			}
		} catch (Throwable ex) {
			log.error("Error while processing incoming event", ex);
		}
    }

	private void sendDeviceMessage(final DBDevice device, final DeviceMessageType deviceMessageType) {
		sendDeviceMessage(device, deviceMessageType, null, null);
    }

	private void sendDeviceMessage(final DBDevice device, final DeviceMessageType deviceMessageType, final Double data1, final Double data2) {
        final DBMessage message = new DBMessage();
        message.setDeviceId(device.getId());
        message.setMessageType(deviceMessageType);
        message.setTimestamp(System.currentTimeMillis());
        message.setBatteryStatus(BatteryStatus.NO_CONDITION_DETECTED);

		if (data1 != null) {
			message.setData1(data1);
		}
		if (data2 != null) {
			message.setData2(data2);
		}
        
		final Event event = EventBuilder.createDeviceMessageEvent(message);
		
        eventAdmin.postEvent(event);
    }

    private Long getDeviceId(final String deviceMAC) {
    	if (isValidMacAddress(deviceMAC)) {
    		final String tempDeviceMAC = deviceMAC.replaceAll("[^a-fA-F0-9]", "");
    		final String hexID = tempDeviceMAC.substring(tempDeviceMAC.length() - 6);
    		
	    	try {
	    		return Long.parseLong(hexID, 16);
	    	} catch (NumberFormatException ex) {
	    		//
	    	}
	    }
    	return null;
    }

    private Long getPortId(final Long deviceId, final Integer port) {
    	if (netIO.isPortValid(port)) {
    		return (deviceId * 10) + port;
    	}
    	return null;
    }

	private boolean isValidMacAddress(final String mac) {
		final Pattern patternMacPairs = Pattern.compile("^([a-fA-F0-9]{2}[:\\.-]?){5}[a-fA-F0-9]{2}$");
		final Pattern patternMacTriples = Pattern.compile("^([a-fA-F0-9]{3}[:\\.-]?){3}[a-fA-F0-9]{3}$");

		// Mac addresses usually are 6 * 2 hex nibbles separated by colons,
		// but apparently it is legal to have 4 * 3 hex nibbles as well,
		// and the separators can be any of : or - or . or nothing.
		return (patternMacPairs.matcher(mac).find() || patternMacTriples.matcher(mac).find());
	}

	/**
	 * Setter of {@link EventAdmin} for blueprint
	 * 
	 * @param eventAdmin The {@link EventAdmin} instance
	 */
	public void setEventAdmin(final EventAdmin eventAdmin) {
		this.eventAdmin = eventAdmin;
	}

	/**
	 * Setter of IpHost for blueprint
	 * 
	 * @param ipHost The host address of NetIO
	 */
	public final void setIpHost(final String ipHost) {
		this.ipHost = PropertyUtil.sanitizeValue(ipHost);
	}

	/**
	 * Setter of IpPort for blueprint
	 * 
	 * @param ipPort The port of NetIO
	 */
	public final void setIpPort(final String ipPort) {
		try {
			this.ipPort = Integer.valueOf(PropertyUtil.sanitizeValue(ipPort));
		} catch (NumberFormatException ex) {
			//
		}
	}

	/**
	 * Setter of Username for blueprint
	 * 
	 * @param username The username for accessing NetIO
	 */
	public final void setUsername(final String username) {
		this.username = PropertyUtil.sanitizeValue(username);
	}

	/**
	 * Setter of Password for blueprint
	 * 
	 * @param password The password for accessing NetIO
	 */
	public final void setPassword(final String password) {
		this.password = PropertyUtil.sanitizeValue(password);
	}

}
