/*
 * Copyright 2016 crea-doo.at
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License. 
 */
package at.creadoo.homer.decanter.collector;

import java.net.InetAddress;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.log4j.Logger;
import org.osgi.service.event.Event;
import org.osgi.service.event.EventAdmin;
import org.osgi.service.event.EventConstants;
import org.osgi.service.event.EventHandler;

import at.ac.ait.hbs.homer.core.common.DataAccess;
import at.ac.ait.hbs.homer.core.common.configuration.ConfigurationService;
import at.ac.ait.hbs.homer.core.common.configuration.ConfigurationValueKey;
import at.ac.ait.hbs.homer.core.common.device.DeviceValue;
import at.ac.ait.hbs.homer.core.common.enumerations.DeviceMessageType;
import at.ac.ait.hbs.homer.core.common.event.EventProperties;
import at.ac.ait.hbs.homer.core.common.event.EventTopic;
import at.ac.ait.hbs.homer.core.common.event.util.EventTopicMatcher;
import at.ac.ait.hbs.homer.core.common.event.util.EventUtil;
import at.ac.ait.hbs.homer.core.common.model.DBDevice;
import at.ac.ait.hbs.homer.core.common.model.DBFlat;
import at.ac.ait.hbs.homer.core.common.model.DBMessage;
import at.ac.ait.hbs.homer.core.common.model.DBRoom;
import at.ac.ait.hbs.homer.core.common.scenario.SystemState;
import at.ac.ait.hbs.homer.core.common.systemstate.SystemStateBlackboard;
import at.ac.ait.hbs.homer.core.common.systemstate.SystemStateListener;

public final class Collector implements EventHandler, SystemStateListener {

	private static final Logger log = Logger.getLogger(Collector.class);
    
    private static final List<String> excludeTopics = new ArrayList<String>();
	
	private static final String COLLECTOR_FILTER_ALL = "*";
	
	private static final String DECANTER_TOPIC_PREFIX = "decanter/";
	
	private static final String COLLECTOR_DECANTER_TOPIC_PREFIX = "decanter/collect/homer/";

    private ConfigurationService configurationService;
    
    private DataAccess dataAccess;
    
    private EventAdmin eventAdmin;

	private SystemStateBlackboard systemStateBlackboard;
	
	private List<String> topics = new ArrayList<String>();

	private final Map<Integer, DBDevice> deviceCache = new HashMap<Integer, DBDevice>();

	private final Map<Integer, DBRoom> roomCache = new HashMap<Integer, DBRoom>();

	private final Map<Integer, DBFlat> flatCache = new HashMap<Integer, DBFlat>();
	
	private String homerSystemID = "";
    
    static {
    	// Filter out service events (started, stopped,...) and log events
    	excludeTopics.add("org/osgi/service/");
    	// Filter out bundle events (started, stopped,...)
    	excludeTopics.add("org/osgi/framework/");
    	// Filter out framework events (features, shell,...)
    	excludeTopics.add("org/apache/karaf/");
    	// Filter out all decanter events
    	excludeTopics.add(DECANTER_TOPIC_PREFIX);
    }
	
	/**
	 * Init method called by container
	 */
	public void init() {
		log.info("Starting " + this.getClass().getSimpleName());
		
		try {
			homerSystemID = configurationService.getMetaValue(ConfigurationValueKey.SYSTEM_ID).getStringValue();
		} catch (Throwable ex) {
			//
		}
		
		// Register listener
		systemStateBlackboard.addListener(this);
		
		updateCache();
	}

	/**
	 * Destroy method called by container. Shut down gracefully
	 */
	public void destroy() {
		log.info("Stopping " + this.getClass().getSimpleName());
		
		// Unregister listener
		systemStateBlackboard.removeListener(this);
	}

	/**
	 * Set the reference to the {@link SystemStateBlackboard} instance
	 * 
	 * @param blackBoard
	 */
	public final void setBlackBoard(final SystemStateBlackboard blackBoard) {
		this.systemStateBlackboard = blackBoard;
	}
	
	/**
	 * Set the reference to the {@link ConfigurationService} instance
	 * 
	 * @param configurationService
	 */
    public final void setConfigurationService(final ConfigurationService configurationService) {
        this.configurationService = configurationService;
    }
	
	/**
	 * Set the reference to the {@link DataAccess} instance
	 * 
	 * @param dataAccess
	 */
    public final void setDataAccess(final DataAccess dataAccess) {
        this.dataAccess = dataAccess;
    }
	
	/**
	 * Set the reference to the {@link EventAdmin} instance
	 * 
	 * @param eventAdmin
	 */
    public final void setEventAdmin(final EventAdmin eventAdmin) {
        this.eventAdmin = eventAdmin;
    }
	
	/**
	 * Set the topics that should be handled
	 * 
	 * @param topics
	 */
	public final void setTopics(final String topics) {
		this.topics = Arrays.asList(topics.split(","));
	}

	@Override
	public void handleEvent(final Event event) {
		if (event == null) {
			return;
		}
		
		final String topic = event.getTopic();
		
		// Handle internal used messages first
		if (EventUtil.isEventOfType(event, EventTopic.HOMECONTROL_DEVICES_CONFIGURATION_CHANGE_MESSAGE)) {
			log.debug("Reloading device cache");
			updateCache();
		}
		
		// Check against excluded topics
		for (final String excludeTopic : excludeTopics) {
			if (topic.startsWith(excludeTopic)) {
				return;
			}
		}

		// Check against included topics
		boolean resume = false;
		for (String item : topics) {
			if (COLLECTOR_FILTER_ALL.equals(item.trim())) {
				resume = true;
				break; 
			}
			
			if (EventTopicMatcher.isMatch(topic, item.trim())) {
				resume = true;
				break;
			}
		}
		
		// If nothing matched abort
		if (!resume) {
			return;
		}
		
        final Map<String, Object> properties = new HashMap<>();
        enrichGeneral(properties);

        // Handle HOMER events
        if (EventUtil.isEventComplete(event)) {
        	properties.put("topic", topic);
        	properties.put(EventConstants.EVENT_TOPIC, topic);
    		try {
    			if (EventUtil.isEventOfType(event, EventTopic.HOMECONTROL_DEVICE_MESSAGE) || EventUtil.isEventOfType(event, EventTopic.HOMECONTROL_DEVICE_STATUS_MESSAGE)) {
    				final DBMessage message = (DBMessage) event.getProperty(EventProperties.MESSAGE.name());
    				final DeviceValue deviceValue = (DeviceValue) event.getProperty(EventProperties.DEVICE_VALUE.name());
    				
    				enrichDeviceValue(properties, deviceValue);
    				
					properties.put("messageID", message.getDeviceId());
					properties.put("messageType", message.getMessageType().name());
					properties.put("messageTypeTitle", message.getMessageType().getTitle());
					properties.put("messageTypeID", message.getMessageType().getId());
					
					properties.put("batteryStatus", message.getBatteryStatus().name());
					properties.put("batteryStatusID", message.getBatteryStatus().getValue());
					
					properties.put("messageTimestamp", message.getTimestamp().getMillis());
					properties.put("messageTimestampReadable", message.getTimestamp().toString());

					if (deviceCache.containsKey(message.getDeviceId())) {
						final DBDevice device = deviceCache.get(message.getDeviceId());
						enrichDevice(properties, device);
					}

					if (event.containsProperty(EventProperties.TAG.name())) {
			        	properties.put("tag", event.getProperty(EventProperties.TAG.name()));
					}
    			} else if (EventUtil.isEventOfType(event, EventTopic.HOMECONTROL_DEVICE_REQUEST)) {
    				final DBDevice device = (DBDevice) event.getProperty(EventProperties.DEVICE.name());
    				final DeviceMessageType deviceMessageType = (DeviceMessageType) event.getProperty(EventProperties.DEVICE_MESSAGE_TYPE.name());
    				final DeviceValue deviceValue = (DeviceValue) event.getProperty(EventProperties.DEVICE_VALUE.name());

					enrichDevice(properties, device);
    				enrichDeviceValue(properties, deviceValue);
					
					properties.put("deviceMessageType", deviceMessageType.name());
					properties.put("deviceMessageTypeTitle", deviceMessageType.getTitle());
					properties.put("deviceMessageTypeID", deviceMessageType.getId());
					
    			} else if (EventUtil.isEventOfType(event, EventTopic.HOMECONTROL_DEVICE_STATUS_REQUEST)) {
    				final DBDevice device = (DBDevice) event.getProperty(EventProperties.DEVICE.name());

					enrichDevice(properties, device);

					if (event.containsProperty(EventProperties.TAG.name())) {
			        	properties.put("tag", event.getProperty(EventProperties.TAG.name()));
					}
    			}
    		} catch (Throwable ex) {
    			log.error("Error while processing incoming event", ex);
    		}
        } else {
	        for (String property : event.getPropertyNames()) {
	            if (property.equals("type")) {
	                if (event.getProperty(property) != null) {
	                    properties.put("eventType", event.getProperty(property).toString());
	                } else {
	                    properties.put("eventType", "eventadmin");
	                }
	            } else {
	                properties.put(property, event.getProperty(property));
	            }
	        }
        }
        eventAdmin.sendEvent(new Event(COLLECTOR_DECANTER_TOPIC_PREFIX + topic, properties));
	}
	
	@Override
	public void handleSystemStateSet(final SystemState state, final SystemState stateOld) {
		if (state != null) {
	        final Map<String, Object> properties = new HashMap<>();
	        enrichGeneral(properties);

        	properties.put("topic", EventTopic.SYSTEM_MESSAGE.getValue());
        	properties.put(EventConstants.EVENT_TOPIC, EventTopic.SYSTEM_MESSAGE.getValue());
			
	        properties.put("stateID", state.getId());
	        properties.put("stateName", state.getName());
	        properties.put("stateDisplayable", state.getDisplayable());
	        properties.put("statePublishable", state.getPublishable());
	        properties.put("stateType", state.getType().name());
	        properties.put("stateTypeTitle", state.getType().getTitle());
	        properties.put("stateValue", state.getValue());
	        properties.put("stateDefaultValue", state.getDefaultValue());
	        if (stateOld != null) {
	        	properties.put("stateOldValue", stateOld.getValue());
	        }
			
	        eventAdmin.sendEvent(new Event(COLLECTOR_DECANTER_TOPIC_PREFIX + EventTopic.SYSTEM_MESSAGE.getValue(), properties));
		}
	}

	@Override
	public void handleSystemStateRemove(final SystemState state) {
		//
	}

    private void updateCache() {
    	deviceCache.clear();
    	roomCache.clear();
    	flatCache.clear();
    	
    	if (dataAccess == null) {
    		return;
    	}
    	
    	deviceCache.putAll(dataAccess.getDevicesAsMap());
		log.debug("Devices in cache: " + deviceCache.size());
    	
		roomCache.putAll(dataAccess.getRoomsAsMap());
		log.debug("Rooms in cache: " + roomCache.size());
    	
		flatCache.putAll(dataAccess.getFlatsAsMap());
		log.debug("Flats in cache: " + flatCache.size());
    }
	
    private void enrichGeneral(final Map<String, Object> properties) {
        properties.put("type", "homer");
        properties.put("systemID", homerSystemID);
        
        String karafName = System.getProperty("karaf.name");
        if (karafName != null) {
            properties.put("karafName", karafName);
        }
        try {
            properties.put("hostAddress", InetAddress.getLocalHost().getHostAddress());
            properties.put("hostName", InetAddress.getLocalHost().getHostName());
        } catch (Exception e) {
            // nothing to do
        }
    }
    
	private void enrichDevice(final Map<String, Object> properties, final DBDevice device) {
		if (device == null) {
			return;
		}
		
		properties.put("deviceID", device.getId());
		
		properties.put("deviceType", device.getType().name());
		properties.put("deviceTypeTitle", device.getType().getTitle());
		properties.put("deviceTypeID", device.getType().getValue());

		properties.put("deviceCategoryType", device.getCategoryType().name());
		properties.put("deviceCategoryTypeID", device.getCategoryType().getValue());
		
		properties.put("deviceProtocol", device.getProtocol().name());
		properties.put("deviceProtocolTitle", device.getProtocol().getTitle());
		properties.put("deviceProtocolID", device.getProtocol().getValue());

		if (device.getLocX() != null) {
			properties.put("deviceLocationX", device.getLocX());
		} else {
			properties.put("deviceLocationX", 0d);
		}
		if (device.getLocY() != null) {
			properties.put("deviceLocationY", device.getLocY());
		} else {
			properties.put("deviceLocationY", 0d);
		}
		if (device.getLocZ() != null) {
			properties.put("deviceLocationZ", device.getLocZ());
		} else {
			properties.put("deviceLocationZ", 0d);
		}

		if (roomCache.containsKey(device.getRoomId())) {
			final DBRoom room = roomCache.get(device.getRoomId());
			enrichRoom(properties, room);
		}

		if (flatCache.containsKey(device.getFlatId())) {
			final DBFlat flat = flatCache.get(device.getFlatId());
			enrichFlat(properties, flat);
		}
		
		for (String key : device.getMetaValues().keySet()) {
			try {
				properties.put(key, device.getMetaValue(key).getValue().toString());
			} catch (Throwable ex) {
				//
			}
		}
	}
	
	private void enrichDeviceValue(final Map<String, Object> properties, final DeviceValue deviceValue) {
		if (deviceValue == null) {
			return;
		}
		
		properties.put("deviceValue", deviceValue.getValue().toString());
		properties.put("deviceValueType", deviceValue.getType().name());
	}

	private void enrichRoom(final Map<String, Object> properties, final DBRoom room) {
		if (room == null) {
			return;
		}
		
		properties.put("roomID", room.getId());
		
		properties.put("roomType", room.getType().name());
		properties.put("roomTypeTitle", room.getType().getTitle());
		properties.put("roomTypeID", room.getType().getValue());

		if (room.getHeight() != null) {
			properties.put("roomHeight", room.getHeight());
		} else {
			properties.put("roomHeight", 0d);
		}
		
		for (String key : room.getMetaValues().keySet()) {
			try {
				properties.put(key, room.getMetaValue(key).getValue().toString());
			} catch (Throwable ex) {
				//
			}
		}
	}

	private void enrichFlat(final Map<String, Object> properties, final DBFlat flat) {
		if (flat == null) {
			return;
		}
		
		properties.put("flatID", flat.getId());

		properties.put("flatAddress", flat.getAddress());
		properties.put("flatDescription", flat.getDescription());

		for (String key : flat.getMetaValues().keySet()) {
			try {
				properties.put(key, flat.getMetaValue(key).getValue().toString());
			} catch (Throwable ex) {
				//
			}
		}
	}

}
