package at.borkowski.spicej.proxy;

import java.math.BigDecimal;
import java.math.RoundingMode;

import at.borkowski.spicej.ticks.TickSource;

/**
 * A utility class for calculating parameters necessary for rate limitation of
 * bytes per seconds.
 * 
 * The way spiceJ works internally, three parameters are required to set up a
 * byte-rate limited stream:
 * 
 * <ul>
 * <li>Tick interval: How many nanoseconds are between two ticks (see
 * {@link TickSource})</li>
 * <li>Bytes per tick: How many bytes are allowed to be transmitted/received per
 * tick</li>
 * <li>Prescale: By how much to divide the incoming ticks (
 * <code>1 - prescale</code> ticks are skipped before actually allowing data to
 * be sent or received). This is a possibility of skipping ticks to allow for
 * very low bitrates without having extensive tick intervals.</li>
 * </ul>
 * 
 * This class allows for easy calculation of bytes per second to these three
 * parametes while respecting some bounds:
 * 
 * <ul>
 * <li>Minimum interval: The interval must not be shorter than
 * {@link #MIN_INTERVAL_NS} nanoseconds to avoid overhead by CPU usage.</li>
 * <li>Maximum interval: An upper bound must not be exceeded for the tick
 * interval, {@link #MAX_INTERVAL_NS} ns.</li>
 * </ul>
 * 
 * The error generated by this class is influenced by the scale used in
 * calculations (see {@link #SCALE}) is tested by
 * <code>RateCalculatorTest</code> (in test sources).
 * 
 */
public class RateCalculator {

   private RateCalculator() {}

   /**
    * The minimum byte rate representable with our data model (this is not
    * necessarily the exact minimum value, but a value close enough to the
    * minimum lower bound.
    */
   public static final float MIN_RATE = 0.00000000094F;

   /**
    * The highest byte rate representable with our data model (this is not
    * necessarily the exact maximum value, but a value close enough to the
    * maximum upper bound.
    */
   public static final float MAX_RATE = 340092346638528859811704183484516925440F;

   /**
    * Nanoseconds per second.
    */
   public static final int NS_PER_S = 1000000000;

   /**
    * Nanoseconds per millisecond.
    */
   public static final int NS_PER_MS = 1000000;

   /**
    * The lower bound for the interval. Intervals lower than this might not be
    * reliably possible on some hardware.
    */
   public static final int MIN_INTERVAL_NS = 50 * NS_PER_MS;

   /**
    * The upper bound for the interval.
    */
   public static final int MAX_INTERVAL_NS = NS_PER_S; // 1 second = maximal interval

   /**
    * The threshold value for byterates above which MIN_INTERVAL is reached.
    */
   public static final long THRESHOLD = (long) Math.ceil((double) NS_PER_S / MIN_INTERVAL_NS);

   /**
    * The scale (see {@link BigDecimal}) used for internal calculations.
    */
   public static final int SCALE = 60;

   /**
    * Performs a calculation with the default minimal interval (see
    * {@link #MIN_INTERVAL_NS}). See {@link RateCalculator} for details.
    * 
    * @param rateBytesPerSecond
    *           The rate in bytes per second to target
    * @return the calculation result
    */
   public static Result calculate(float rateBytesPerSecond) {
      return calculate(rateBytesPerSecond, MIN_INTERVAL_NS);
   }

   /**
    * Performs a calculation with the given minimal interval. See
    * {@link RateCalculator} for details.
    * 
    * @param rateBytesPerSecond
    *           The rate in bytes per second to target
    * @param minInterval
    *           The minimal interval in nanoseconds.
    * @return the calculation result
    */
   public static Result calculate(float rateBytesPerSecond, int minInterval) {
      int threshold = (int) Math.ceil((double) NS_PER_S / minInterval);

      if (rateBytesPerSecond < 1) {
         BigDecimal interval_ = new BigDecimal(NS_PER_S).divide(new BigDecimal(rateBytesPerSecond), SCALE, RoundingMode.HALF_UP);

         while (interval_.compareTo(new BigDecimal(MAX_INTERVAL_NS)) > 0)
            interval_ = interval_.divide(new BigDecimal(2));

         // current rate: 1 [b] / interval [s]
         // prescale = current rate / rate
         //          = 1 [b] / interval / rate

         // interval = min(1 / rate, MAX_INTERVAL_NS)

         // proof that prescale will never be < 1:

         // condition for prescale < 1:
         // 1/(interval * rate) < 1     | * interval
         // 1 / rate < interval         | [A]

         // if LS of min: interval = 1 / rate, then 1 = interval * rate (-> prescale = 1)
         // if RS of min: interval = MAX_INTERVAL_NS
         //    then interval < 1 / rate -> contradiction to [A]

         // int prescale = (int) Math.round((double) 1D / interval / rateBytesPerSecond * NS_PER_S);         
         BigDecimal prescale = new BigDecimal(1);
         prescale = prescale.divide(interval_, SCALE, RoundingMode.HALF_UP);
         prescale = prescale.divide(new BigDecimal(rateBytesPerSecond), SCALE, RoundingMode.HALF_UP);
         prescale = prescale.multiply(new BigDecimal(NS_PER_S));

         if (prescale.compareTo(new BigDecimal(Integer.MAX_VALUE)) > 0)
            throw new IllegalArgumentException("rate too low (necessary prescale too big)");

         return new Result(1, prescale.intValue(), interval_.intValue());
      } else if (rateBytesPerSecond < threshold) {
         // 1 <= rate < MAX_FRACTIONAL_RATE
         // interval: 1s / rate

         int interval = (int) (NS_PER_S / rateBytesPerSecond);
         return new Result(1, 1, interval);
      } else {
         int interval = (int) minInterval;
         BigDecimal bytespertick = new BigDecimal(rateBytesPerSecond);
         bytespertick = bytespertick.divide(new BigDecimal(NS_PER_S), SCALE, RoundingMode.HALF_UP);
         bytespertick = bytespertick.multiply(new BigDecimal(interval));
         return new Result(bytespertick.intValue(), 1, interval);
      }
   }

   public static class Result {
      private final int bytesPerTick;
      private final int prescale;
      private final int tickNanosecondInterval;

      public Result(int bytesPerTick, int prescale, int tickNanosecondInterval) {
         this.bytesPerTick = bytesPerTick;
         this.prescale = prescale;
         this.tickNanosecondInterval = tickNanosecondInterval;
      }

      public int getBytesPerTick() {
         return bytesPerTick;
      }

      public int getPrescale() {
         return prescale;
      }

      public int getTickNanosecondInterval() {
         return tickNanosecondInterval;
      }
   }
}
