package com.apppartner.core.views;

/*
 * ----------------------------------IMPORTANT NOTE-----------------------------------------
 * To Gain Access to the CustomAttributes found in attrs.xml, such as 'typeface'
 * you must include the following in your layout's xml file in the parent element
 *
 * xmlns:apppartnerlib="http://schemas.android.com/apk/res-auto"
 *
 * then you can use the extra elements in this way
 * apppartnerlib:typeface="customFont.ttf"
 * -------------------------------------------------------------------------------------------
 */

import android.content.Context;
import android.content.res.TypedArray;
import android.graphics.Paint;
import android.graphics.Typeface;
import android.support.v4.util.LruCache;
import android.util.AttributeSet;
import android.widget.EditText;

import com.apppartner.core.R;

/**
 * Allows you to define a custom font type in its layout XML
 * @author Jake Green
 */
public class CustomEditText extends EditText
{
    //================================================================================
    // Class Properties
    //================================================================================

    private static LruCache<String, Typeface> sTypefaceCache = new LruCache<String, Typeface>(12);

    //================================================================================
    // LifeCycle Methods
    //================================================================================

    public CustomEditText(Context context)
    {
        super(context);
        init(null);
    }

    public CustomEditText(Context context, AttributeSet attrs)
    {
        super(context, attrs);
        init(attrs);
    }

    public CustomEditText(Context context, AttributeSet attrs, int defStyle)
    {
        super(context, attrs, defStyle);
        init(attrs);
    }

    //================================================================================
    // Private Methods
    //================================================================================

    private void init(final AttributeSet attrs)
    {
        // Get the custom attribute we defined int values/attrs.xml
        if (attrs != null)
        {
            TypedArray a = getContext().obtainStyledAttributes(attrs, R.styleable.CustomEditText);
            try
            {
                String typefaceName = a.getString(R.styleable.CustomEditText_font);

                if (typefaceName != null && !"".equals(typefaceName) && !this.isInEditMode())
                {
                    Typeface typeface = sTypefaceCache.get(typefaceName);

                    if (typeface == null)
                    {
                        typeface = Typeface.createFromAsset(this.getContext().getAssets(), "fonts/" + typefaceName);

                        // Cache the Typeface object so we aren't reloading it from the assets all the time
                        sTypefaceCache.put(typefaceName, typeface);
                    }

                    this.setTypeface(typeface);
                    setPaintFlags(getPaintFlags() | Paint.SUBPIXEL_TEXT_FLAG);
                }
            }
            finally
            {
                a.recycle();
            }
        }
    }
}
