package com.apppartner.core.utils;

import android.content.ContentResolver;
import android.content.Context;
import android.content.Intent;
import android.database.Cursor;
import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.graphics.Canvas;
import android.graphics.Matrix;
import android.media.ExifInterface;
import android.net.Uri;
import android.os.Environment;
import android.provider.MediaStore;
import android.util.DisplayMetrics;
import android.util.Log;
import android.view.View;
import android.view.ViewGroup;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;

/**
 * A class for static helper methods related to bitmaps.
 * <p/>
 * Created on 5/6/16.
 *
 * @author Jake Green
 */
public class BitmapUtils
{
    private static final String TAG = "BitmapUtils";
    private static final int MAX_IMAGE_SIZE = 512;

    public static Bitmap getBitmapFromView(View view, DisplayMetrics displayMetrics)
    {
        view.setLayoutParams(new ViewGroup.LayoutParams(ViewGroup.LayoutParams.WRAP_CONTENT, ViewGroup.LayoutParams.WRAP_CONTENT));
        view.measure(displayMetrics.widthPixels, displayMetrics.heightPixels);
        view.layout(0, 0, displayMetrics.widthPixels, displayMetrics.heightPixels);
        view.buildDrawingCache();

        Bitmap bitmap = Bitmap.createBitmap(view.getMeasuredWidth(), view.getMeasuredHeight(), Bitmap.Config.ARGB_8888);
        Canvas canvas = new Canvas(bitmap);
        view.draw(canvas);

        return bitmap;
    }

    public static Bitmap downsizeBitmap(Uri selectedImage, ContentResolver contentResolver, int maxSize) throws FileNotFoundException
    {
        if (contentResolver == null)
            return null;

        // Decode image size without actually loading the image
        BitmapFactory.Options options = new BitmapFactory.Options();
        options.inJustDecodeBounds = true; // this flag allows us to figure out the dimensions of the image without actually loading it into memory
        BitmapFactory.decodeStream(contentResolver.openInputStream(selectedImage), null, options);

        // Calculate the inSampleSize to scale image down before we load it into memory
        options.inSampleSize = calculateInSampleSize(options, maxSize);
        options.inJustDecodeBounds = false;

        return BitmapFactory.decodeStream(contentResolver.openInputStream(selectedImage), null, options);
    }

    public static Bitmap downsizeBitmap(String filePath, ContentResolver contentResolver, int maxSize) throws FileNotFoundException
    {
        if (contentResolver == null)
            return null;

        // Decode image size without actually loading the image
        BitmapFactory.Options options = new BitmapFactory.Options();
        options.inJustDecodeBounds = true; // this flag allows us to figure out the dimensions of the image without actually loading it into memory
        BitmapFactory.decodeFile(filePath, options);

        // Calculate the inSampleSize to scale image down before we load it into memory
        options.inSampleSize = calculateInSampleSize(options, maxSize);
        options.inJustDecodeBounds = false;

        return BitmapFactory.decodeFile(filePath, options);
    }

    private static int calculateInSampleSize(BitmapFactory.Options options, int maxSize)
    {
        // Raw height and width of image
        final int height = options.outHeight;
        final int width = options.outWidth;
        int inSampleSize = 1;

        if (height > maxSize || width > maxSize)
        {

            final int halfHeight = height / 2;
            final int halfWidth = width / 2;

            // Calculate the largest inSampleSize value that is a power of 2 and keeps both
            // height and width larger than the requested height and width.
            while ((halfHeight / inSampleSize) > maxSize && (halfWidth / inSampleSize) > maxSize)
            {
                inSampleSize *= 2;
            }
        }

        return inSampleSize;
    }

    public static Bitmap rotateBitmapIfNecessary(String picturePath, Bitmap normalBitmap, ContentResolver contentResolver) throws IOException
    {
        Bitmap rotatedBitmap = null;
        int orientation = 0;

        // Find out what the orientation of the bitmap is
        ExifInterface ei = new ExifInterface(picturePath);
        orientation = ei.getAttributeInt(ExifInterface.TAG_ORIENTATION, ExifInterface.ORIENTATION_NORMAL);

        // Rotate the bitmap depending on the orientation
        if (normalBitmap != null)
        {
            switch (orientation)
            {
                case ExifInterface.ORIENTATION_UNDEFINED:
                    break;
                case ExifInterface.ORIENTATION_ROTATE_90:
                    rotatedBitmap = BitmapUtils.rotateBitmap(normalBitmap, 90);
                    break;
                case ExifInterface.ORIENTATION_ROTATE_180:
                    rotatedBitmap = BitmapUtils.rotateBitmap(normalBitmap, 180);
                    break;
                case ExifInterface.ORIENTATION_ROTATE_270:
                    rotatedBitmap = BitmapUtils.rotateBitmap(normalBitmap, 270);
                    break;
            }
        }

        // Return the rotated bitmap, free up the old bitmap
        if (rotatedBitmap != null)
        {
            normalBitmap.recycle();
            normalBitmap = null;

            return rotatedBitmap;
        }

        return normalBitmap;
    }

    public static Bitmap rotateBitmap(Bitmap source, float angle)
    {
        Matrix matrix = new Matrix();
        matrix.postRotate(angle);
        return Bitmap.createBitmap(source, 0, 0, source.getWidth(), source.getHeight(), matrix, true);
    }

    public static String getRealPathFromURI(Uri contentUri, Context context)
    {
        String imagePath = null;

        try
        {
            String[] filePathColumn = {MediaStore.Images.Media.DATA};

            Cursor cursor = context.getContentResolver().query(contentUri, filePathColumn, null, null, null);
            cursor.moveToFirst();

            int columnIndex = cursor.getColumnIndex(filePathColumn[0]);
            imagePath = cursor.getString(columnIndex);

            cursor.close();
            cursor = null;
        }
        catch (Exception e)
        {
            return contentUri.getPath();
        }

        return imagePath;
    }

    public static Bitmap getBitmapFromGalleryData(Intent data, Context context)
    {
        // Get the path of the image that was selected from the gallery
        Uri selectedImage = data.getData();
        String[] filePathColumn = {MediaStore.Images.Media.DATA};

        Cursor cursor = context.getContentResolver().query(selectedImage, filePathColumn, null, null, null);
        cursor.moveToFirst();

        int columnIndex = cursor.getColumnIndex(filePathColumn[0]);
        String picturePath = cursor.getString(columnIndex);

        cursor.close();
        cursor = null;

        try
        {
            // Load the image from the gallery and rotate it to the correct orientation
            final ContentResolver contentResolver = context.getContentResolver();
            Bitmap normalBitmap = BitmapUtils.downsizeBitmap(picturePath, contentResolver, MAX_IMAGE_SIZE);

            return rotateBitmapIfNecessary(picturePath, normalBitmap, contentResolver);
        }
        catch (Exception e)
        {
            Log.w(TAG, e.toString());
        }

        return null;
    }

    public static Bitmap getBitmapFromCameraURI(Context context, Uri cameraImageUri, ContentResolver contentResolver)
    {
        try
        {
            Bitmap downsizedBitmap = BitmapUtils.downsizeBitmap(cameraImageUri, contentResolver, MAX_IMAGE_SIZE);
            String realPath = BitmapUtils.getRealPathFromURI(cameraImageUri, context);
            return BitmapUtils.rotateBitmapIfNecessary(realPath, downsizedBitmap, contentResolver);
        }
        catch (Exception e)
        {
            Log.w(TAG, e.toString());
        }

        return null;
    }

    public static Bitmap getBitmapFromPath(Context context, String picturePath)
    {
        try
        {
            // Load the image from the path and rotate it to the correct orientation
            final ContentResolver contentResolver = context.getContentResolver();
            Bitmap normalBitmap = BitmapUtils.downsizeBitmap(picturePath, contentResolver, MAX_IMAGE_SIZE);

            return rotateBitmapIfNecessary(picturePath, normalBitmap, contentResolver);
        }
        catch (Exception e)
        {
            Log.w(TAG, e.toString());
        }

        return null;
    }

    public static int dpToPx(Context context, int dp)
    {
        DisplayMetrics displayMetrics = context.getResources().getDisplayMetrics();
        int px = Math.round(dp * (displayMetrics.xdpi / DisplayMetrics.DENSITY_DEFAULT));
        return px;
    }

    public static File saveBitmapToFile(Bitmap bitmap, String filename)
    {
        String extStorageDirectory = Environment.getExternalStorageDirectory().toString();
        OutputStream outStream = null;

        File file = new File(extStorageDirectory, filename + ".jpeg");
        if (file.exists())
        {
            file.delete();
            file = new File(extStorageDirectory, filename + ".jpeg");
        }
        try
        {
            outStream = new FileOutputStream(file);
            bitmap.compress(Bitmap.CompressFormat.JPEG, 100, outStream);
            outStream.flush();
            outStream.close();
        }
        catch (Exception e)
        {
            e.printStackTrace();
        }
        return file;
    }
}
