package com.apppartner.core.statelistview;

import android.content.Context;
import android.content.res.TypedArray;
import android.support.annotation.ColorRes;
import android.support.v4.widget.SwipeRefreshLayout;
import android.support.v7.widget.LinearLayoutManager;
import android.support.v7.widget.RecyclerView;
import android.util.AttributeSet;
import android.util.Log;
import android.view.View;
import android.widget.FrameLayout;
import android.widget.LinearLayout;
import android.widget.ProgressBar;

import com.apppartner.core.ContentStates;
import com.apppartner.core.R;

/**
 * Created by Jonathan Muller on 5/6/16.
 *
 * This view is meant for displaying a list of data in a recyclerview. It has the added benefit
 * of having error, empty, and loading views that can be shown to the user.
 *
 * If no views are supplied for the error and empty states a default one will be used
 *
 * To set the error and empty states provide layout resources to in xml using the
 * "empty_state_view" and "error_state_view" attributes
 *
 */
public class StateListView extends LinearLayout implements ContentStates
{
    private static final String TAG = "StateListView";

    // Encapsulates the recyclerview to allow refreshing
    // This must be manually turned on by called enableSwipeToRefresh(true)
    private SwipeRefreshLayout contentRefreshLayout;

    // Layout containing the progress bar
    private View progressLayout;
    // The progress bar that is shown when showLoading() is called
    private ProgressBar progressBar;

    // Container layouts to fill with empty and error state views
    private FrameLayout emptyStateContainer;
    private FrameLayout errorStateContainer;

    // Is used to hold the list
    private RecyclerView recyclerView;

    // Hold references to the empty and error state views
    private View emptyView;
    private View errorView;

    // References to the empty and error states Controllable interfaces
    // These are passed back in getErrorStateView and getEmptyStateView if those views implement
    // the Controllable interface
    private Controllable emptyStateController;
    private Controllable errorStateController;

    public StateListView(Context context)
    {
        super(context);
        init(context, null);
    }

    public StateListView(Context context, AttributeSet attrs)
    {
        super(context, attrs);
        init(context, attrs);
    }

    public StateListView(Context context, AttributeSet attrs, int defStyleAttr)
    {
        super(context, attrs, defStyleAttr);
        init(context, attrs);
    }

    private void init(Context context, AttributeSet attrs)
    {
        View view = inflate(getContext(), R.layout.layout_state_list_view, this);

        recyclerView = (RecyclerView) view.findViewById(R.id.recyclerView);

        contentRefreshLayout = (SwipeRefreshLayout) view.findViewById(R.id.content_refresh_layout);

        progressLayout = view.findViewById(R.id.loading_layout);
        progressBar = (ProgressBar) view.findViewById(R.id.progressBar);

        emptyStateContainer = (FrameLayout) view.findViewById(R.id.empty_state_container);
        errorStateContainer = (FrameLayout) view.findViewById(R.id.error_state_container);

        TypedArray attributes = context.obtainStyledAttributes(attrs, R.styleable.StateListView);

        setupStates(attributes);

        recyclerView.setLayoutManager(new LinearLayoutManager(getContext()));

        contentRefreshLayout.setEnabled(false);
    }

    private void setupStates(TypedArray attributes)
    {
        // Gets the empty state layout id to inflate
        int emptyStateLayoutId = attributes.getResourceId(R.styleable.StateListView_empty_state_view, R.layout.default_empty_state_view);

        // Check if layout is not the default
        if (emptyStateLayoutId != R.layout.default_empty_state_view)
        {
            // inflate the custom layout view
            emptyView = inflate(getContext(), emptyStateLayoutId, null);
            // add it to this view within the emptyStateContainer
            emptyStateContainer.addView(emptyView);

            // If this custom view implements Controllable we get a reference to it
            if (emptyView instanceof Controllable)
            {
                emptyStateController = (Controllable)emptyView;
            }
        }
        else
        {
            // We are using the default layout view, this implements Controllable
            emptyStateController = new EmptyStateView(getContext());
            emptyStateContainer.addView((View) emptyStateController);
        }

        int errorStateLayoutId = attributes.getResourceId(R.styleable.StateListView_error_state_view, R.layout.default_error_state_view);

        if (errorStateLayoutId != R.layout.default_error_state_view)
        {
            errorView = inflate(getContext(), errorStateLayoutId, null);
            errorStateContainer.addView(errorView);

            if (errorView instanceof Controllable)
            {
                errorStateController = (Controllable)errorView;
            }
        }
        else
        {
            errorStateController = new ErrorStateView(getContext());
            errorStateContainer.addView((View) errorStateController);
        }
    }

    /**
     * Sets the colors on the refresh view
     * @param colorRes the colors to show
     */
    public void setRefreshColors(@ColorRes int... colorRes)
    {
        contentRefreshLayout.setColorSchemeResources(colorRes);
    }

    /**
     * Sets the OnRefreshLayouts OnRefreshListener
     * @param onRefreshListener
     */
    public void setOnRefreshListener(SwipeRefreshLayout.OnRefreshListener onRefreshListener)
    {
        contentRefreshLayout.setOnRefreshListener(onRefreshListener);
    }

    /**
     * Sets the background color for the whole view
     * @param colorRes
     */
    public void setContentBackgroundColor(@ColorRes int colorRes)
    {
        contentRefreshLayout.setBackgroundColor(getResources().getColor(colorRes));
    }

    /**
     * Sets the background color for the progressLayout
     * @param colorRes
     */
    public void setLoadingBackgroundColor(@ColorRes int colorRes)
    {
        progressLayout.setBackgroundColor(getResources().getColor(colorRes));
    }

    /**
     * Sets the background color for the emptyStateContainer
     * @param colorRes
     */
    public void setEmptyBackgroundColor(@ColorRes int colorRes)
    {
        emptyStateContainer.setBackgroundColor(getResources().getColor(colorRes));
    }

    /**
     * Sets the background color for the errorStateContainer
     * @param colorRes
     */
    public void setErrorBackgroundColor(@ColorRes int colorRes)
    {
        errorStateContainer.setBackgroundColor(getResources().getColor(colorRes));
    }

    /**
     * Set to true if you want to enable swipe to refresh
     * If enable you must call setOnFreshListener
     * @param enabled
     */
    public void enableSwipeToRefresh(boolean enabled)
    {
        contentRefreshLayout.setEnabled(enabled);
    }

    public void setRefresh(boolean on)
    {
        contentRefreshLayout.setEnabled(on);
    }

    /**
     * Set the message that is shown in the empty state
     * @param message
     */
    public void setEmptyStateMessage(String message)
    {
        if (emptyStateController != null)
        {
            emptyStateController.setMessage(message);
        }
        else
        {
            Log.e(TAG, "Can not set message, empty state view doesn't implement Controllable");
        }
    }

    /**
     * Set the image that is shown in the empty state
     * @param drawableRes
     */
    public void setEmptyStateImage(int drawableRes)
    {
        if (emptyStateController != null)
        {
            emptyStateController.setImage(drawableRes);
        }
        else
        {
            Log.e(TAG, "Can not set image, empty state view doesn't implement Controllable");
        }
    }

    /**
     * Set the click listener for the empty state.
     * The default views will call this when the message text is clicked
     * @param onClickListener
     */
    public void setEmptyStateOnClick(OnClickListener onClickListener)
    {
        if (emptyStateController != null)
        {
            emptyStateController.setClickListener(onClickListener);
        }
        else
        {
            Log.e(TAG, "Can not set a click listener, empty state view doesn't implement Controllable");
        }
    }

    /**
     * Set the message that is shown in the error state
     * @param message
     */
    public void setErrorStateMessage(String message)
    {
        if (errorStateController != null)
        {
            errorStateController.setMessage(message);
        }
        else
        {
            Log.e(TAG, "Can not set message, empty state view doesn't implement Controllable");
        }
    }

    /**
     * Set the image that is shown in the error state
     * @param drawableRes
     */
    public void setErrorStateImage(int drawableRes)
    {
        if (errorStateController != null)
        {
            errorStateController.setImage(drawableRes);
        }
        else
        {
            Log.e(TAG, "Can not set image, empty state view doesn't implement Controllable");
        }
    }

    /**
     * Set the click listener for the error state.
     * The default views will call this when the message text is clicked
     * @param onClickListener
     */
    public void setErrorStateOnClick(OnClickListener onClickListener)
    {
        if (errorStateController != null)
        {
            errorStateController.setClickListener(onClickListener);
        }
        else
        {
            Log.e(TAG, "Can not set a click listener, empty state view doesn't implement Controllable");
        }
    }

    /**
     * Set the empty state view.
     * This will replace any empty state view that was there before
     * If the view implements Controllable it can be controlled
     * like a default view
     * @param view
     */
    public void setEmptyStateView(View view)
    {
        // If there is already a view, remove it
        if (emptyStateContainer.getChildCount() != 0)
        {
            emptyStateContainer.removeAllViews();
        }

        emptyStateContainer.addView(view);

        if (view instanceof Controllable)
        {
            emptyStateController = (Controllable)view;
        }
    }

    /**
     * Set the error state view.
     * This will replace any error state view that was there before
     * If the view implements Controllable it can be controlled
     * like a default view
     * @param view
     */
    public void setErrorStateView(View view)
    {
        // If there is already a view, remove it
        if (errorStateContainer.getChildCount() != 0)
        {
            errorStateContainer.removeAllViews();
        }

        errorStateContainer.addView(view);

        if (view instanceof Controllable)
        {
            errorStateController = (Controllable)view;
        }
    }

    //=================================================================================
    // ContentStates Implementation
    //=================================================================================

    @Override
    public void showLoading()
    {
        progressLayout.setVisibility(VISIBLE);
        contentRefreshLayout.setVisibility(GONE);
        emptyStateContainer.setVisibility(GONE);
        errorStateContainer.setVisibility(GONE);
    }

    @Override
    public void showContent()
    {
        progressLayout.setVisibility(GONE);
        contentRefreshLayout.setVisibility(VISIBLE);
        emptyStateContainer.setVisibility(GONE);
        errorStateContainer.setVisibility(GONE);
    }

    @Override
    public void showEmpty()
    {
        progressLayout.setVisibility(GONE);
        contentRefreshLayout.setVisibility(GONE);
        emptyStateContainer.setVisibility(VISIBLE);
        errorStateContainer.setVisibility(GONE);
    }

    @Override
    public void showError()
    {
        progressLayout.setVisibility(GONE);
        contentRefreshLayout.setVisibility(GONE);
        emptyStateContainer.setVisibility(GONE);
        errorStateContainer.setVisibility(VISIBLE);
    }

    public void setAdapter(RecyclerView.Adapter adapter)
    {
        recyclerView.setAdapter(adapter);
    }

    public RecyclerView getRecyclerView()
    {
        return recyclerView;
    }

    public View getEmptyStateView()
    {
        if (emptyStateController != null)
        {
            return (View) emptyStateController;
        }
        return emptyView;
    }

    public View getErrorStateView()
    {
        if (errorStateController != null)
        {
            return (View) errorStateController;
        }
        return errorView;
    }
}