package app.visly.io

import app.visly.CheckNewVersionError
import app.visly.Config
import app.visly.DownloadAssetsError
import app.visly.io.type.Platform
import com.apollographql.apollo.ApolloCall
import com.apollographql.apollo.ApolloClient
import com.apollographql.apollo.api.Operation
import com.apollographql.apollo.api.Query
import com.apollographql.apollo.api.Response
import com.apollographql.apollo.api.cache.http.HttpCachePolicy
import com.apollographql.apollo.exception.ApolloException
import kotlinx.coroutines.runBlocking
import kotlin.coroutines.resume
import kotlin.coroutines.suspendCoroutine

class Client {

    fun fetchLatestVersion(): String {
        val result = query(
            query =
            LatestVersionQuery.builder().build(),
            serverUrl = Config.SERVER_URL
        )

        return result.data?.let { data ->
            return data.codegenTools.android.latestVersion
        }
        ?: throw result.error?.let { error ->
            (error as? GraphQlError)?.let {
                CheckNewVersionError.graphQlError(
                    description = it.description,
                    userMessage = "Could not fetch latest plugin version."
                )
            } ?: throw CheckNewVersionError.networkError("Network call failed")
        }
        ?: Throwable("Unknown error")
    }

    fun fetchAssets(apiToken: String, version: String, serverUrl: String): AssetsQuery.Data {

        val result = query(
            query =
            AssetsQuery.builder()
                .apiToken(apiToken)
                .platform(Platform.ANDROID)
                .version(version)
                .build(),
            serverUrl = serverUrl
        )

        return result.data?.let { data ->
            return data
        }
        ?: throw result.error?.let { error ->
            (error as? GraphQlError)?.let {
                DownloadAssetsError.graphQlError(
                    description = it.description,
                    userMessage = "Could not fetch assets. Is api key correct?"
                )
            } ?: throw DownloadAssetsError.networkError("Network call failed")
        }
        ?: Throwable("Unknown error")
    }

    private fun <D : Operation.Data, T, V : Operation.Variables>
            query(query: Query<D, T, V>, serverUrl: String): QueryResult<T> = runBlocking {

            suspendCoroutine<QueryResult<T>> { continuation ->
                ApolloClient
                    .builder()
                    .serverUrl("$serverUrl/graphql")
                    .defaultHttpCachePolicy(HttpCachePolicy.NETWORK_ONLY)
                    .build()
                    .query(query)
                    .enqueue(object : ApolloCall.Callback<T>() {

                        override fun onFailure(e: ApolloException) {
                            continuation.resume(QueryResult(error = e))
                        }

                        override fun onResponse(response: Response<T>) {
                            continuation.resume(
                                QueryResult(
                                    data = response.data(),
                                    error = response.errors().fold("") { message, error -> String
                                        val errorMessage = error.message()?.let { it } ?: "Unknown error"
                                        "$message\n$errorMessage"
                                    }.let { description ->
                                        GraphQlError(description = description)
                                    }
                                )
                            )
                        }
                    })
            }
        }

    companion object {
        class GraphQlError(val description: String) : Throwable(description)
        data class QueryResult<T>(val data: T? = null, val error: Throwable? = null)
    }
}