package app.visly.filesystem

import java.io.File
import java.nio.charset.Charset

data class FileSystemConfiguration(
    val homeDir: String,
    val projectDir: String,
    val outputDir: String
)

enum class FilePath {
    HOME_DIR,
    PROJECT_DIR,
    OUTPUT_DIR
}

class FileSystemError(message: String) : Throwable(message)

interface FileOperations {
    fun delete(path: String): Boolean
    fun deleteRecursively(path: String): Boolean
    fun mkdir(path: String): Boolean
    fun mkdirs(path: String): Boolean
    fun exists(path: String): Boolean
    fun readText(path: String, charset: Charset): String
    fun writeText(text: String, path: String, charset: Charset)
    fun writeBytes(bytes: ByteArray, path: String)
    fun copy(fromPath: String, toPath: String): Boolean
}

class DefaultFileOperations : FileOperations {

    override fun delete(path: String): Boolean {
        return File(path).delete()
    }

    override fun deleteRecursively(path: String): Boolean {
        return File(path).deleteRecursively()
    }

    override fun mkdir(path: String): Boolean {
        return File(path).mkdir()
    }

    override fun mkdirs(path: String): Boolean {
        return File(path).mkdirs()
    }

    override fun exists(path: String): Boolean {
        return File(path).exists()
    }

    override fun readText(path: String, charset: Charset): String {
        return File(path).readText(charset = charset)
    }

    override fun writeText(text: String, path: String, charset: Charset) {
        File(path).writeText(text = text, charset = charset)
    }

    override fun writeBytes(bytes: ByteArray, path: String) {
        File(path).writeBytes(bytes)
    }

    override fun copy(fromPath: String, toPath: String): Boolean {
        return File(fromPath).copyRecursively(File(toPath), overwrite = false)
    }
}

private fun assertMatches(path: String, pattern: String) {
    if (!Regex(pattern).matches(path)) {
        throw FileSystemError("Path $path is not valid for file system")
    }
}

private fun assertValidFileSystem(fileSystem: FileSystem) {
    assertMatches(fileSystem.path(FilePath.OUTPUT_DIR), "^/.*/src/visly/gen")
    assertMatches(fileSystem.path(FilePath.PROJECT_DIR), "^.*/VislyAndroid")
    assertMatches(fileSystem.path(FilePath.HOME_DIR), "^/.*/.visly")
}

private fun projectRootDir(): String {
    val appId = SystemUtil.currentDir().replace("/", "_")
    return "${SystemUtil.tempDir()}/app.visly/$appId/VislyAndroid"
}

class FileSystem internal constructor(
    configuration: FileSystemConfiguration,
    private val paths: Map<FilePath, String> = mapOf(
        FilePath.HOME_DIR to configuration.homeDir,
        FilePath.PROJECT_DIR to configuration.projectDir,
        FilePath.OUTPUT_DIR to configuration.outputDir
    ),
    private val operations: FileOperations = DefaultFileOperations()
) : FileOperations {

    init {
        assertValidFileSystem(this)
    }

    fun path(key: FilePath): String {
        return paths.get(key)?.let { it } ?: error("Path with key: $key should be in file system")
    }

    fun isInSystem(path: String): Boolean {
        val normalizedPath = File(path).normalize().absolutePath
        paths.values.forEach {
            if (normalizedPath == it) {
                return true
            }

            if (normalizedPath.startsWith("$it/")) {
                return true
            }
        }
        return false
    }

    fun assertInSystem(path: String) {
        if (!isInSystem(path)) {
            throw FileSystemError("Path $path is not in visly file system")
        }
    }

    override fun delete(path: String): Boolean {
        assertInSystem(path)
        return operations.delete(path)
    }

    override fun deleteRecursively(path: String): Boolean {
        assertInSystem(path)
        return operations.deleteRecursively(path)
    }

    override fun mkdir(path: String): Boolean {
        assertInSystem(path)
        return operations.mkdir(path)
    }

    override fun mkdirs(path: String): Boolean {
        assertInSystem(path)
        return operations.mkdirs(path)
    }

    override fun exists(path: String): Boolean {
        assertInSystem(path)
        return operations.exists(path)
    }

    override fun readText(path: String, charset: Charset): String {
        assertInSystem(path)
        return operations.readText(path = path, charset = charset)
    }

    override fun writeText(text: String, path: String, charset: Charset) {
        assertInSystem(path)
        operations.writeText(text = text, path = path, charset = charset)
    }

    override fun writeBytes(bytes: ByteArray, path: String) {
        assertInSystem(path)
        operations.writeBytes(bytes, path)
    }

    override fun copy(fromPath: String, toPath: String): Boolean {
        assertInSystem(toPath)
        return operations.copy(fromPath, toPath)
    }

    companion object {

        lateinit var instance: FileSystem

        fun setup(projectDir: String) {
            instance = FileSystem(
                configuration = FileSystemConfiguration(
                    homeDir = "${SystemUtil.userHomeDir()}/.visly",
                    projectDir = projectRootDir(),
                    outputDir = "$projectDir/src/visly/gen"
                )
            )
        }
    }

    override fun toString(): String {
        return """
        FileSystem:
        outputDir: ${path(FilePath.OUTPUT_DIR)}
        projectDir: ${path(FilePath.PROJECT_DIR)}
        homeDir: ${path(FilePath.HOME_DIR)}
        """
    }
}

fun FileSystem.Companion.projectDir(): String {
    return FileSystem.instance.path(FilePath.PROJECT_DIR)
}

fun FileSystem.Companion.outputDir(): String {
    return FileSystem.instance.path(FilePath.OUTPUT_DIR)
}

fun FileSystem.Companion.homeDir(): String {
    return FileSystem.instance.path(FilePath.HOME_DIR)
}