package app.visly.codegen.textstyle

import app.visly.CodegenError
import app.visly.codegen.AssetGenerator
import app.visly.codegen.AssetUtil
import app.visly.codegen.Release
import app.visly.codegen.util.*
import app.visly.filesystem.VislyFS
import app.visly.io.AssetsQuery
import com.commit451.resourcespoet.ResourcesPoet
import com.commit451.resourcespoet.StyleItem

class TextStylesGenerator(private val resDir: String, private val downloader: FileDownloader) : AssetGenerator {

    override fun generateAssets(release: Release) {
        VislyFS().mkdirs("$resDir/font")
        VislyFS().mkdirs("$resDir/styles")

        val resPoet = ResourcesPoet.create(true)

        release.textStyles().forEach {
            generateTextStyle(it, resPoet)
        }

        val xml = resPoet.build().toByteArray(Charsets.UTF_8)
        VislyFS().writeBytes(xml, "$resDir/values/styles.xml")
    }

    private fun generateTextStyle(style: AssetsQuery.TextStyle, resPoet: ResourcesPoet) {
        val name = "${AssetUtil.resource_prefix}${style.name().toValidResourceName()}"
        val font = style.font()

        val fontName = AssetUtil.resource_prefix +
                "${font.family().toValidResourceName()}_" +
                "${font.fontWeight().toValidResourceName()}_" +
                font.fontStyle().toString().toLowerCase().toValidResourceName()

        val items = listOf(
            StyleItem("android:fontFamily", "@font/$fontName"),
            StyleItem("android:textSize", "${style.fontSize().value().vmlPx().dp.xmlFormatted()}sp"),
            StyleItem("android:lineSpacingMultiplier", "0"),
            StyleItem("android:lineSpacingExtra", "${style.lineHeight().value().vmlPx().dp.xmlFormatted()}dp"))

        val files: ByteArray
        try {
            files = downloader.download(font.fileUrl())
        } catch (error: Throwable) {
            throw CodegenError.textStylesError(
                description = "Failed to download font: ${font.fileUrl()}. ${error.message}",
                userMessage = "Failed to download font: ${font.fileUrl()}")
        }

        createFile(files, fontName, font.fileExtension())
        resPoet.addStyle(name, "TextAppearance.AppCompat", items)
    }

    private fun createFile(data: ByteArray, name: String, extension: String) {
        VislyFS().writeBytes(
            data,
            "$resDir/font/$name.$extension"
        )
    }
}
