package app.visly.codegen.image

import app.visly.CodegenError
import app.visly.VislyLog
import app.visly.codegen.AssetGenerator
import app.visly.codegen.AssetUtil
import app.visly.codegen.Release
import app.visly.codegen.util.*
import app.visly.filesystem.VislyFS
import app.visly.io.AssetsQuery

class ImagesGenerator(private val resDir: String) : AssetGenerator {

    override fun generateAssets(release: Release) {
        VislyFS().mkdirs("$resDir/drawable")
        release.images().forEach {
            generateImage(it)
        }
    }

    private fun generateImage(image: AssetsQuery.Image) {
        assert(image.versions().size > 0)
        val mediaType = image.versions().first().mimetype()
        val imageName = "${AssetUtil.resource_prefix}${image.name().toValidResourceName()}"
        when (mediaType) {
            MediaType.IMG_SVG_XML.type -> {
                generateVectorDrawable(image, imageName)
                VislyLog.d("Generated icon: $imageName")
            }
            MediaType.IMG_PNG.type -> {
                generatePngs(image, imageName)
                VislyLog.d("Generated icon: $imageName")
            }
            else -> throw CodegenError.iconsError("Image ${image.name()} uses unsupported media type: $mediaType. Perhaps upgrade Visly plugin?")
        }
    }

    private fun generateVectorDrawable(image: AssetsQuery.Image, name: String) {
        val svgDataUri = image.versions().first().base64()
        val svgData = svgDataUri.dataUrlBase64()?.base64DecodeString() ?: throw CodegenError.iconsError("Could not decode base64 from: $svgDataUri")
        val data = SVGParser().parse(svgData, image.name())
        createFile(data.toByteArray(Charsets.UTF_8), "$resDir/drawable/$name.xml")
    }

    private fun generatePngs(image: AssetsQuery.Image, name: String) {
        for (version in image.versions()) {
            val pngDataUri = version.base64()
            val pngData = pngDataUri.dataUrlBase64()?.base64Decode() ?: throw CodegenError.iconsError("Could not decode base64 from: $pngDataUri")
            val dir = "$resDir/${DrawableHelper.drawableFolderForDensity(version.density())}"
            if (!VislyFS().exists(dir)) {
                VislyFS().mkdirs(dir)
            }
            val path = "$dir/$name.png"
            createFile(pngData, path)
        }
    }

    private fun createFile(data: ByteArray, path: String) {
        VislyFS().writeBytes(
            data,
            path)
    }
}
