package app.visly

import app.visly.codegen.AssetsGenerator
import app.visly.codegen.color.ColorsGenerator
import app.visly.codegen.image.ImagesGenerator
import app.visly.codegen.string.StringsGenerator
import app.visly.codegen.textstyle.TextStylesGenerator
import app.visly.codegen.util.URLFileDownloader
import app.visly.io.Client
import app.visly.project.ProjectInstaller
import app.visly.version.VersionChecker
import org.gradle.api.GradleException

// Wrapper for tracking errors thrown from Visly commands
class VislyWrapper(private val visly: Visly) : Visly {

    override fun install(apiKey: String, version: String, serverUrl: String) {
        track {
            visly.install(apiKey, version, serverUrl)
        }
    }

    override fun shouldCheckNewVersion(): Boolean {
        return track {
            visly.shouldCheckNewVersion()
        }
    }

    override fun checkNewVersion() {
        track {
            visly.checkNewVersion()
        }
    }

    private fun <T> track(closure: () -> T): T {
        try {
            return closure()
        } catch (error: VislyError) {
            VislyLog.track(error)
            throw GradleException(
                """
                    ${error.message}
                    
                    Having problems? 😥 
                    Feel free to contact mailto:help@visly.app for assistance! 🤗
                    
                """.trimIndent()
            )
        }
    }
}

interface Visly {
    fun install(apiKey: String, version: String, serverUrl: String)
    fun shouldCheckNewVersion(): Boolean
    fun checkNewVersion()
}

open class VislyImpl(
    val config: VislyConfiguration,
    private val assetsGenerator: AssetsGenerator = AssetsGenerator(
        config.projectDir,
        listOf(
            ColorsGenerator(config.projectResDir),
            ImagesGenerator(config.projectResDir),
            StringsGenerator(config.projectResDir),
            TextStylesGenerator(config.projectResDir, URLFileDownloader())
        )
    ),
    private val projectInstaller: ProjectInstaller = ProjectInstaller(
        projectResDir = config.projectResDir,
        outputDir = config.outputDir,
        outputResDir = config.outputResDir
    ),
    private val client: Client = Client(),
    private val versionChecker: VersionChecker = VersionChecker(
        client = client,
        lastCheckFile = config.versionLastCheckFile,
        currentVersion = Config.VERSION,
        checkIntervalInMillis = config.versionCheckIntervalInMillis
    ),
    logger: Logger? = null
) : Visly {

    init { logger?.let { VislyLog.set(it) } }

    override fun install(apiKey: String, version: String, serverUrl: String) {
        VislyLog.track("Install started")

        VislyLog.d("Api-key: '$apiKey'")
        VislyLog.d("Release: '$version'")
        VislyLog.d("Server url: '$serverUrl'")
        VislyLog.lc("Syncing project...")

        VislyLog.lc("Fetching release version: '$version'")
        val release = client.fetchAssets(apiKey, version, serverUrl).team().releaseInfo()

        VislyLog.lc("Generating assets...")
        assetsGenerator.process(release)

        VislyLog.lc("Installing project...")
        projectInstaller.install()

        VislyLog.lc("Successfully installed!")
        VislyLog.lc("Visly resources stored in: ${config.outputDir}")

        VislyLog.track("Install successful")
    }

    override fun shouldCheckNewVersion(): Boolean {
        return versionChecker.shouldCheckNewVersion()
    }

    override fun checkNewVersion() {
        VislyLog.lc("Current plugin version: ${Config.VERSION}")
        VislyLog.lc("Checking for newer version...")
        versionChecker.checkNewVersion()?.let { newVersion ->
            VislyLog.lc("**********************************************")
            VislyLog.lc("A newer plugin version is available! -> $newVersion")
            VislyLog.lc("**********************************************")
        } ?: VislyLog.lc("Plugin is up-to-date.")
    }
}
