package app.visly.codegen.textstyle

import app.visly.codegen.AssetGenerator
import app.visly.codegen.AssetUtil
import app.visly.codegen.Release
import app.visly.codegen.util.ColorUtil
import app.visly.codegen.util.FileDownloader
import app.visly.codegen.util.convertDimension
import app.visly.codegen.util.toValidResourceName
import app.visly.io.AssetsQuery
import com.commit451.resourcespoet.ResourcesPoet
import com.commit451.resourcespoet.StyleItem
import java.io.File

class TextStylesGenerator(private val resDir: String, private val downloader: FileDownloader) : AssetGenerator {

    override fun generateAssets(release: Release) {
        File("$resDir/font").mkdirs()
        File("$resDir/styles").mkdirs()

        val resPoet = ResourcesPoet.create(true)

        release.textStyles().forEach {
            generateTextStyle(it, resPoet)
        }

        val xml = resPoet.build().toByteArray(Charsets.UTF_8)
        val file = File("$resDir/values/styles.xml")
        file.createNewFile()
        file.writeBytes(xml)
    }

    private fun generateTextStyle(style: AssetsQuery.TextStyle, resPoet: ResourcesPoet) {
        val name = "${AssetUtil.resource_prefix}${style.name().toValidResourceName()}"
        val font = style.font()

        val fontName = AssetUtil.resource_prefix +
                "${font.family().toValidResourceName()}_" +
                "${font.fontWeight().toValidResourceName()}_" +
                font.fontStyle().toString().toLowerCase().toValidResourceName()

        val fontColor = ColorUtil.rgbaToHexString(
            style.fontColor().red(),
            style.fontColor().green(),
            style.fontColor().blue(),
            style.fontColor().alpha())

        val items = listOf(
            StyleItem("android:fontFamily", "@font/$fontName"),
            StyleItem("android:textColor", fontColor),
            StyleItem("android:textSize", "${convertDimension(style.fontSize().value())}sp"),
            StyleItem("android:lineSpacingMultiplier", "0"),
            StyleItem("android:lineSpacingExtra", "${convertDimension(style.lineHeight().value())}dp"))

        createFile(downloader.download(font.fileUrl()), fontName, font.fileExtension())
        resPoet.addStyle(name, "TextAppearance.AppCompat", items)
    }

    private fun createFile(data: ByteArray, name: String, extension: String) {
        val file = File("$resDir/font/$name.$extension")
        file.createNewFile()
        file.writeBytes(data)
    }
}
