package app.visly.codegen.image

import app.visly.VislyLog
import app.visly.codegen.AssetGenerator
import app.visly.codegen.AssetUtil
import app.visly.codegen.Release
import app.visly.codegen.util.*
import app.visly.io.AssetsQuery
import java.io.File

class ImagesGenerator(private val resDir: String) : AssetGenerator {

    override fun generateAssets(release: Release) {
        File("$resDir/drawable").mkdirs()
        release.icons().forEach {
            generateImage(it)
        }
    }

    private fun generateImage(icon: AssetsQuery.Icon) {
        val image = icon.base64()
        val mediaType = image.dataUrlMediaType() ?: throw VislyException("Could not get mime type from: $image")
        val dataBase64 = image.dataUrlBase64() ?: throw VislyException("Could not get data segment type: $image")
        val data = dataBase64.base64Decode() ?: throw VislyException("Could not decode base64 from: $image")
        when (mediaType) {
            MediaType.IMG_SVG_XML.type -> {
                val name = "${AssetUtil.resource_prefix}${icon.name().toValidResourceName()}"
                generateSVG(data, name)
                VislyLog.d("Generated icon: $name")
            }
            else -> throw VislyException("Image $icon.name() uses unsupported media type: $mediaType. Perhaps upgrade Visly plugin?")
        }
    }

    private fun generateSVG(svg: String, name: String) {
        val data = SVGParser().parse(svg, name)
        createFile(data.toByteArray(Charsets.UTF_8), name)
    }

    private fun createFile(data: ByteArray, name: String) {
        val file = File("$resDir/drawable/$name.xml")
        file.createNewFile()
        file.writeBytes(data)
    }
}
