package app.visly.io

import com.apollographql.apollo.ApolloCall
import com.apollographql.apollo.ApolloClient
import com.apollographql.apollo.api.Response
import com.apollographql.apollo.api.cache.http.HttpCachePolicy
import com.apollographql.apollo.exception.ApolloException
import org.gradle.api.GradleException
import java.util.concurrent.CountDownLatch

class Client() {

    enum class Server(val url: String) {
        PROD("https://api.visly.app/graphql"),
        LOCAL("http://api.localhost:3000/graphql")
    }

    private fun fetchAssetsAsync(apiToken: String, version: String, callback: Callback<AssetsQuery.Data>) {
        ApolloClient
            .builder()
            .serverUrl(Server.PROD.url)
            .defaultHttpCachePolicy(HttpCachePolicy.NETWORK_ONLY)
            .build()
            .query(
                AssetsQuery.builder()
                    .apiToken(apiToken)
                    .version(version)
                    .build()
            ).enqueue(object : ApolloCall.Callback<AssetsQuery.Data>() {
                override fun onFailure(e: ApolloException) {
                    callback.onError(e)
                }

                override fun onResponse(response: Response<AssetsQuery.Data>) {
                    callback.onResult(response.data())
                }
            })
    }

    fun fetchAssets(apiToken: String, version: String): AssetsQuery.Data {
        var fetchResult: AssetsQuery.Data? = null
        val latch = CountDownLatch(1)
        try {
            fetchAssetsAsync(apiToken, version, object : Callback<AssetsQuery.Data> {
                override fun onResult(result: AssetsQuery.Data?) {
                    fetchResult = result
                    latch.countDown()
                }

                override fun onError(error: Throwable) {
                    latch.countDown()
                    throw GradleException("Could not fetch assets: ${error.localizedMessage}")
                }
            })
        } catch (error: Exception) {
            latch.countDown()
            throw error
        }

        while (latch.count != 0L) {
            Thread.sleep(200)
        }

        return fetchResult.let { it } ?: throw GradleException("Could not fetch assets.")
    }

    companion object {
        interface Callback<T> {
            fun onResult(result: T?)
            fun onError(error: Throwable)
        }
    }
}