/**
 * Copyright (c) Visly Inc.
 *
 * This source code is licensed under the MIT license found in the
 * LICENSE file in the root directory of this source tree.
 */

package app.visly.shard.viewimpl

import android.content.Context
import android.view.View
import android.view.ViewGroup
import android.widget.HorizontalScrollView
import android.widget.ScrollView
import androidx.recyclerview.widget.LinearLayoutManager
import androidx.recyclerview.widget.PagerSnapHelper
import androidx.recyclerview.widget.RecyclerView
import app.visly.shard.*
import kotlin.math.max

internal enum class Direction {
    VERTICAL,
    HORIZONTAL,
}

internal class ShardViewHolder(view: ShardRootView): RecyclerView.ViewHolder(view) {
    val root: ShardRootView = view
}

internal class ShardRecyclerAdapter(
        val context: Context,
        val direction: Direction,
        val items: List<ShardRoot>,
        val paginated: Boolean
): RecyclerView.Adapter<ShardViewHolder>() {
    override fun onCreateViewHolder(parent: ViewGroup, viewType: Int): ShardViewHolder {
        val root = ShardRootView(context)
        root.layoutParams = if (paginated) {
            RecyclerView.LayoutParams(
                    RecyclerView.LayoutParams.MATCH_PARENT,
                    RecyclerView.LayoutParams.MATCH_PARENT
            )
        } else if (direction == Direction.HORIZONTAL) {
            RecyclerView.LayoutParams(
                    RecyclerView.LayoutParams.WRAP_CONTENT,
                    RecyclerView.LayoutParams.MATCH_PARENT
            )
        } else {
            RecyclerView.LayoutParams(
                    RecyclerView.LayoutParams.MATCH_PARENT,
                    RecyclerView.LayoutParams.WRAP_CONTENT
            )
        }
        return ShardViewHolder(root)
    }

    override fun getItemCount(): Int {
        return items.size
    }

    override fun onBindViewHolder(holder: ShardViewHolder, position: Int) {
        holder.root.setRoot(items[position])
    }
}

class ScrollViewImpl(ctx: ShardContext): BaseViewImpl<RecyclerView>(ctx) {
    internal var direction = Direction.VERTICAL
    internal var contentInset = 0
    internal var paginated = false
    internal var children: List<ShardRoot> = emptyList()

    override fun measure(width: Float?, height: Float?): Size {
        var measuredWidth = 0.0f
        var measuredHeight = 0.0f

        for (child in children) {
            val size = child.measure(width, height)

            if (paginated) {
                measuredWidth = max(measuredWidth, size.width)
                measuredHeight = max(measuredHeight, size.height)
            } else if (direction == Direction.VERTICAL) {
                measuredHeight += size.height
                measuredWidth = max(measuredWidth, size.width)
            } else {
                measuredWidth += size.width
                measuredHeight = max(measuredHeight, size.height)
            }
        }

        return Size(width ?: measuredWidth, height ?: measuredHeight)
    }

    override fun setProp(key: String, value: JsonValue) {
        super.setProp(key, value)

        when (key) {
            "direction" -> {
                direction = when (value) {
                    JsonValue.String("horizontal") -> Direction.HORIZONTAL
                    JsonValue.String("vertical") -> Direction.VERTICAL
                    else -> Direction.VERTICAL
                }
            }

            "paginated" -> {
                paginated = when (value) {
                    is JsonValue.Boolean -> true
                    else  -> false
                }
            }

            "content-inset" -> {
                contentInset = when (value) {
                    is JsonValue.Object -> value.toDips(ctx).toInt()
                    else -> 0
                }
            }

            "children" -> {
                children = when (value) {
                    is JsonValue.Array -> {
                        value.value.map {
                            val content = ShardViewManager.instance.loadJson(ctx, it)
                            if (content.isError()) {
                                throw content.error()
                            }
                            content.success()
                        }
                    }
                    else -> emptyList()
                }
            }
        }
    }

    override fun createView(): RecyclerView {
        val view = RecyclerView(ctx)
        view.clipToPadding = false
        view.isHorizontalScrollBarEnabled = false
        view.isVerticalScrollBarEnabled = false
        return view
    }

    override fun bindView(view: RecyclerView) {
        super.bindView(view)

        view.setPadding(
                if (direction == Direction.VERTICAL) 0 else contentInset,
                if (direction == Direction.HORIZONTAL) 0 else contentInset,
                if (direction == Direction.VERTICAL) 0 else contentInset,
                if (direction == Direction.HORIZONTAL) 0 else contentInset)

        val recyclerViewDirection = if (direction == Direction.HORIZONTAL) {
            RecyclerView.HORIZONTAL
        } else {
            RecyclerView.VERTICAL
        }

        view.adapter = ShardRecyclerAdapter(ctx, direction, children, paginated)
        view.layoutManager = LinearLayoutManager(ctx, recyclerViewDirection, false)

        if (paginated) {
            val snapHelper = PagerSnapHelper()
            snapHelper.attachToRecyclerView(view)
        }

        PagerSnapHelper()
    }
}