package app.visly.shard

import android.app.Activity
import android.content.Context
import android.content.Intent
import android.net.Uri
import android.os.Bundle
import android.util.Log

// Create a connection to platform.shardlib.com and configure
// your app to show notifications configured on the platform.
//
// To force a notification to show regardless of server settings
// or whether you have seen the notification previously pass `force = true`.
// This is useful to test notifications locally while developing.
//
// In Activity.onCreate() of your main activity insert the following
// code to show users what's new in the current version of the app if it is
// the first time they are running it since updating.
//
// val platform = ShardPlatform(appId = "<APP_ID>")
// platform.showWhatsNew(this, force = BuildConfig.DEBUG)
//
// If you would like to have more control over when the what's new screen is shown
// you may pass an optional onSomethingNew closure. This closure is itself passed a
// closure which when called will show the whats new screen. This API gives you control
// over when to show the screen, perhaps in response to pressing a button in your UI.
//
// platform.showWhatsNew(force = BuildConfig.DEBUG, onSomethingNew = { show ->
//     show()
// })
class ShardPlatform(val appId: String) {

    private fun getAppVersion(ctx: Context): String {
        return ctx.packageManager.getPackageInfo(ctx.packageName, 0).versionName
    }

    private fun buildUrl(ctx: Context, minVersion: String): String {
        return Uri.Builder()
                .scheme("https")
                .authority("api.shardlib.com")
                .path("whats_new")
                .appendQueryParameter("min-version", minVersion)
                .appendQueryParameter("version", getAppVersion(ctx))
                .appendQueryParameter("os", "android")
                .appendQueryParameter("appid", appId)
                .build()
                .toString()
    }

    fun showWhatsNew(
            ctx: Context,
            force: Boolean = false,
            onSomethingNew: (() -> Unit) -> Unit = { show ->
                show()
            }
    ) {
        val prefs = ctx.getSharedPreferences("com.shardlib", Context.MODE_PRIVATE)

        var initialVersion = prefs.getString("initial-version", null)
        if (initialVersion == null) {
            initialVersion = getAppVersion(ctx)
            prefs.edit().putString("initial-version", initialVersion).apply()
        }

        if (force) {
            initialVersion = null
        }

        val start = System.currentTimeMillis()

        // As we cannot pass ShardRoot between activities we have to make sure to pre-load the shard
        // json before starting the activity so we know that the shard is loaded instantly when the
        // activity is presented.
        ShardViewManager.instance.loadShardJsonFromUrl(buildUrl(ctx, initialVersion ?: "0.0.0")) { result ->
            if (result.isError()) {
                return@loadShardJsonFromUrl
            }

            val json = result.success()

            if (!force && (System.currentTimeMillis() - start) > 2000) {
                Log.d("shard",  "Skipping whats new dialog as it took too long to fetch. Will show on next launch.")
                return@loadShardJsonFromUrl
            }

            if (!force) {
                val seenIds = prefs.getStringSet("seen-ids", mutableSetOf()) ?: mutableSetOf()

                val jsonValue = JsonValue.parse(json) as JsonValue.Object
                val shardId = jsonValue.value["id"] as JsonValue.String?

                if (shardId == null || seenIds.contains(shardId.value)) {
                    return@loadShardJsonFromUrl
                } else {
                    seenIds.add(shardId.value)
                    prefs.edit().putStringSet("seen-ids", seenIds).apply()
                }
             }

            onSomethingNew {
                val shardIntent = Intent(ctx, ShardActivity::class.java)
                shardIntent.putExtra("shard-json", json)
                ctx.startActivity(shardIntent)
            }
        }
    }
}

class ShardActivity: Activity() {

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)

        val shard = intent.extras?.getString("shard-json")!!
        val result = ShardViewManager.instance.loadJson(this, shard)

        if (result.isError()) {
            result.error().printStackTrace()
            finish()
            return
        }

        val root = result.success()
        val rootView = ShardRootView(this)

        rootView.on("open-url") {
            val url = (it as JsonValue.String).value
            startActivity(Intent(Intent.ACTION_VIEW, Uri.parse(url)))
        }

        rootView.on("dismiss-alert") {
            finish()
        }

        rootView.on("update") {
            if (it != null && it is JsonValue.String) {
                val playStoreLink = it.value
                val intent = Intent(Intent.ACTION_VIEW).apply {
                    data = Uri.parse(playStoreLink)
                }
                startActivity(intent)
            }
            finish()
        }

        rootView.setRoot(root)
        setContentView(rootView)
    }
}