/**
 * Copyright (c) Visly Inc.
 *
 * This source code is licensed under the MIT license found in the
 * LICENSE file in the root directory of this source tree.
 */

package app.visly.shard

import android.graphics.Typeface
import android.content.res.AssetManager
import android.graphics.Paint
import android.text.TextPaint
import android.text.style.MetricAffectingSpan

data class ShardTextSpanAttributes(
        val style: Int? = null,
        val weight: Int? = null,
        val size: Float? = null,
        val color: Int? = null,
        val fontFamily: String? = null
)

class ShardTextSpan(
    private val attributes: ShardTextSpanAttributes,
    private val ctx: ShardContext): MetricAffectingSpan() {


    override fun updateDrawState(paint: TextPaint) {
        apply(paint, attributes, ctx.resources.assets)
    }

    override fun updateMeasureState(paint: TextPaint) {
        apply(paint, attributes, ctx.resources.assets)
    }

    private fun apply(paint: Paint, attributes: ShardTextSpanAttributes, assetManager: AssetManager) {
        var typeface = paint.getTypeface()
        val oldStyle = typeface?.getStyle() ?: 0

        var want = 0
        if (attributes.weight == Typeface.BOLD || oldStyle and Typeface.BOLD != 0 && attributes.weight == null) {
            want = want or Typeface.BOLD
        }

        if (attributes.style == Typeface.ITALIC || oldStyle and Typeface.ITALIC != 0 && attributes.style == null) {
            want = want or Typeface.ITALIC
        }

        typeface = if (attributes.fontFamily != null) {
            val customFont = ctx.fontCollection[attributes.fontFamily]
            if (customFont != null) {
                Typeface.create(customFont, want)
            } else {
                ShardFontManager.instance.getTypeface(attributes.fontFamily, want, assetManager)
            }
        } else if (typeface != null) {
            Typeface.create(typeface, want)
        } else {
            typeface
        }

        paint.typeface = if (typeface != null) {
            typeface
        } else {
            Typeface.defaultFromStyle(want)
        }

        if (attributes.color != null) paint.color = attributes.color
        if (attributes.size != null) paint.textSize = attributes.size

        paint.isSubpixelText = true
    }

}