package tookan.tookantrack.utils;

import android.app.Activity;
import android.content.Context;
import android.location.Location;
import android.location.LocationManager;
import android.os.Build;
import android.provider.Settings;
import android.support.v4.app.ActivityCompat;
import android.util.Log;

import com.google.android.gms.common.ConnectionResult;
import com.google.android.gms.common.GoogleApiAvailability;
import com.google.android.gms.maps.model.LatLng;

import tookan.tookantrack.R;
import tookan.tookantrack.metering.Keys;
import tookan.tookantrack.metering.utils.Prefs;
import tookan.tookantrack.retrofit2.TrackingCodes;

import static android.content.ContentValues.TAG;
import static tookan.tookantrack.utils.CommonUtils.isPermissionGranted;


public class TrackingLocationUtils {

    private static final int PLAY_SERVICES_RESOLUTION_REQUEST = 1972;
    public static double LATITUDE = 0;
    public static double LONGITUDE = 0;

    /**
     * Method to initialize the Location
     */
    public static void init(Context context) {

        try {

            long latitude = Prefs.with(context).getLong(Keys.Prefs.LATITUDE, 0);
            long longitude = Prefs.with(context).getLong(Keys.Prefs.LONGITUDE, 0);

            LATITUDE = Double.longBitsToDouble(latitude);
            LONGITUDE = Double.longBitsToDouble(longitude);

        } catch (Exception ex) {

            Prefs.with(context).remove(Keys.Prefs.LATITUDE);
            Prefs.with(context).remove(Keys.Prefs.LONGITUDE);

            init(context);
        }
    }

    /**
     * Method to save the com.example.library.tookan.mylibrary.location in preferences
     *
     * @param location
     */
    public static void saveLocation(Context context, Location location) {

        if (location == null)
            return;

        LATITUDE = location.getLatitude();
        LONGITUDE = location.getLongitude();

        Prefs.with(context).save(Keys.Prefs.LATITUDE, Double.doubleToRawLongBits(LATITUDE));
        Prefs.with(context).save(Keys.Prefs.LONGITUDE, Double.doubleToRawLongBits(LONGITUDE));
    }

    /**
     * Method to check whether the mock locations are enabled
     *
     * @return
     */
    public static boolean isMockLocationsEnabled(Context context) {

        return Build.VERSION.SDK_INT < Build.VERSION_CODES.M &&
                !(Settings.Secure.getString(context.getContentResolver(),
                        Settings.Secure.ALLOW_MOCK_LOCATION).equals("0"));
    }

    /**
     * Method to check whether com.example.library.tookan.mylibrary.location services
     * through network are enabled or not
     *
     * @return
     */
    private static boolean isProviderEnabled(Context context, String provider) {

        LocationManager manager = (LocationManager) context.getSystemService(Context.LOCATION_SERVICE);

        try {
            return manager.isProviderEnabled(provider);
        } catch (Exception ex) {
            return false;
        }
    }

    /**
     * Method to check whether the GPS is enabled or not
     *
     * @param context
     * @return
     */
    public static boolean isGPSEnabled(Context context) {

        return isProviderEnabled(context, LocationManager.GPS_PROVIDER);
    }

    /**
     * Method to check whether com.example.library.tookan.mylibrary.location services via Network are enabled or not
     *
     * @return
     */
    public static boolean isNetworkEnabled(Context context) {

        return isProviderEnabled(context, LocationManager.NETWORK_PROVIDER);
    }

    /**
     * Method to save the com.example.library.tookan.mylibrary.location in preferences
     */
    public static Location getLastLocation(Context context) {

        long lat = Prefs.with(context).getLong(Keys.Prefs.LATITUDE, 0);
        long lng = Prefs.with(context).getLong(Keys.Prefs.LONGITUDE, 0);

        double latitude = Double.longBitsToDouble(lat);
        double longitude = Double.longBitsToDouble(lng);

        Location lastLocation = new Location("LAST_LOCATION");

        lastLocation.setLatitude(latitude);
        lastLocation.setLongitude(longitude);

        return lastLocation;
    }

    /**
     * Method to build com.example.library.tookan.mylibrary.location from latitude and Longitude
     *
     * @param latitude
     * @param longitude
     * @return
     */
    public static Location toLocation(double latitude, double longitude) {

        Location location = new Location("Location");
        location.setLatitude(latitude);
        location.setLongitude(longitude);
        return location;
    }

    /**
     * Method to build com.example.library.tookan.mylibrary.location from latitude and Longitude
     *
     * @param locationLatLng
     * @return
     */
    public static Location toLocation(LatLng locationLatLng) {

        if (locationLatLng == null)
            return null;

        return toLocation(locationLatLng.latitude, locationLatLng.longitude);
    }

    public static boolean isPlayServicesAvailable(Activity activity) {
        GoogleApiAvailability googleAPI = GoogleApiAvailability.getInstance();
        int result = googleAPI.isGooglePlayServicesAvailable(activity);
        if (result != ConnectionResult.SUCCESS) {
            if (googleAPI.isUserResolvableError(result)) {
                googleAPI.getErrorDialog(activity, result,
                        PLAY_SERVICES_RESOLUTION_REQUEST).show();
            }

            return false;
        }

        return true;
    }

    public static boolean isUdpEnable(Context context) {
        return Prefs.with(context).getBoolean(Keys.Prefs.IS_UDP_ENABLED, false);
    }

    /**
     * Method to check whether the Location Permission
     * is Granted by the User
     */
    public static boolean checkLocationPermissions(Activity activity) {

        /** Code to check whether the Location Permission is Granted */
        String[] permissionsRequired = new String[]{android.Manifest.permission.ACCESS_COARSE_LOCATION,
                android.Manifest.permission.ACCESS_FINE_LOCATION};

        return askUserToGrantPermission(activity, permissionsRequired,
                activity.getString(R.string.please_grant_permission_location_text), TrackingCodes.Permission.LOCATION);
    }

    /**
     * Method to check whether the Permission is Granted by the User
     * <p/>
     * permission type: DANGEROUS
     *
     * @param activity
     * @param permissions
     * @param explanation
     * @param requestCode
     * @return
     */
    public static boolean askUserToGrantPermission(Activity activity, String[] permissions, String explanation, int requestCode) {

        Log.e(TAG, "askUserToGrantPermission");

        String permissionRequired = null;

        for (String permission : permissions)
            if (!isPermissionGranted(activity, permission)) {
                permissionRequired = permission;
                break;
            }

        // Check if the Permission is ALREADY GRANTED
        if (permissionRequired == null) return true;

        // Check if there is a need to show the PERMISSION DIALOG
        boolean explanationRequired = ActivityCompat.shouldShowRequestPermissionRationale(activity, permissionRequired);
        Log.e(TAG, "askUserToGrantPermission: explanationRequired(" + explanationRequired + "): " + permissionRequired);

        // Convey the EXPLANATION if required
        if (explanationRequired) {

            if (explanation == null) explanation = "Please grant permission";
            CommonUtils.snackBar(activity, explanation, TrackingCodes.SnackBarType.ERROR);
        } else {

            // We can request the permission, if no EXPLANATIONS required
            ActivityCompat.requestPermissions(activity, permissions, requestCode);
        }

        return false;
    }

}
