package tookan.tookantrack.retrofit2;

import android.app.Activity;

import java.lang.ref.WeakReference;
import java.net.ConnectException;
import java.net.SocketTimeoutException;
import java.net.UnknownHostException;

import retrofit2.Call;
import retrofit2.Callback;
import retrofit2.Response;
import tookan.tookantrack.model.TrackingBaseModel;


/**
 * Custom Retrofit TrackingResponseResolver
 *
 * @param <T>
 */
public abstract class TrackingResponseResolver<T extends TrackingBaseModel> implements Callback<T> {
    public final static String UNEXPECTED_ERROR_OCCURRED = "Something went wrong. Please try again later";
    private static final String TAG = TrackingResponseResolver.class.getSimpleName();
    private final String NO_INTERNET_MESSAGE = "No internet connection. Please try again later.";
    private final String REMOTE_SERVER_FAILED_MESSAGE = "Application server could not respond. Please try later.";
    private WeakReference<Activity> weakActivity = null;
    private Boolean showLoading = false;
    private Boolean showError = false;


    /**
     * @param activity
     */
    public TrackingResponseResolver(Activity activity) {
        this.weakActivity = new WeakReference<>(activity);
        this.showError = true;
    }


    /**
     * @param activity
     * @param showLoading
     */
    public TrackingResponseResolver(Activity activity, Boolean showLoading) {
        this.weakActivity = new WeakReference<>(activity);
        this.showLoading = showLoading;
    }


    /**
     * @param activity
     * @param showLoading
     * @param showError
     */
    public TrackingResponseResolver(Activity activity, Boolean showLoading, Boolean showError) {
        this.weakActivity = new WeakReference<>(activity);
        this.showLoading = showLoading;
        this.showError = showError;
    }

    public abstract void success(T t);

    public abstract void failure(TrackingAPIError error);

    @Override
    public void onResponse(Call<T> t, Response<T> tResponse) {
        try {

            resolve(t, tResponse);

        } catch (Throwable e) {
            e.printStackTrace();

            // Pipeline the flow of Control
            //	TrackingAPIError trackingApiError = new TrackingAPIError(PARSING_ERROR.getStatusCode(), TTConstants.StatusDescription.PARSING_ERROR.getMessage(weakActivity.get()));
            //	fireError(trackingApiError);

        }

    }

    @Override
    public void onFailure(Call<T> t, Throwable throwable) {
        fireError(new TrackingAPIError(900, resolveNetworkError(throwable)));
    }


    private void resolve(Call<T> t, Response<T> tResponse) {
        if (tResponse.isSuccessful() && tResponse.body() != null) {

            TrackingBaseModel trackingBaseModel = tResponse.body();

            TrackingCodes.StatusCode status = TrackingCodes.StatusCode.get(trackingBaseModel.getStatus());
            TrackingAPIError trackingApiError = new TrackingAPIError(trackingBaseModel.getStatus(), trackingBaseModel.getMessage());
            switch (status) {

                case ACTION_COMPLETE:
                    success(tResponse.body());
                    break;


                case PARAMETER_MISSING:
                case SHOW_ERROR_MESSAGE:
                    fireError(trackingApiError);
                    break;

                default:
                    fireError(trackingApiError);
                    break;
            }


        }
    }


    /**
     * @param trackingApiError
     */
    public void fireError(TrackingAPIError trackingApiError) {
        failure(trackingApiError);
    }


    /**
     * @param trackingApiError
     * @return
     */
    public Boolean checkAuthorizationError(TrackingAPIError trackingApiError) {
        return true;
    }

    /**
     * Method resolve network errors
     *
     * @param cause
     * @return
     */

    private String resolveNetworkError(Throwable cause) {
        if (cause instanceof UnknownHostException) {
            return NO_INTERNET_MESSAGE;
        } else if (cause instanceof SocketTimeoutException) {
            return REMOTE_SERVER_FAILED_MESSAGE;
        } else if (cause instanceof ConnectException) {
            return REMOTE_SERVER_FAILED_MESSAGE;
        } else if (cause instanceof IllegalStateException) {
            return "Parse Exception";
        }
        return UNEXPECTED_ERROR_OCCURRED;
    }


}
