package tookan.tookantrack.metering;


import android.Manifest;
import android.app.PendingIntent;
import android.content.ContentResolver;
import android.content.Context;
import android.content.Intent;
import android.content.pm.PackageManager;
import android.location.LocationManager;
import android.os.Bundle;
import android.provider.Settings;
import android.support.annotation.NonNull;
import android.support.v4.app.ActivityCompat;

import com.google.android.gms.common.ConnectionResult;
import com.google.android.gms.common.GooglePlayServicesUtil;
import com.google.android.gms.common.api.GoogleApiClient;
import com.google.android.gms.location.LocationRequest;
import com.google.android.gms.location.LocationServices;

import tookan.tookantrack.utils.TTLog;


public class FusedLocationFetcherBackgroundBalanced implements GoogleApiClient.ConnectionCallbacks, GoogleApiClient.OnConnectionFailedListener {

    private static final String TAG = FusedLocationFetcherBackgroundBalanced.class.getSimpleName();

    private static final int LOCATION_PI_ID = 6980;

    private GoogleApiClient googleApiClient;
    private LocationRequest locationrequest;
    private PendingIntent locationIntent;
    private long requestInterval;
    private Context context;


    /**
     * Constructor for initializing LocationFetcher class' object
     *
     * @param context application context
     */
    public FusedLocationFetcherBackgroundBalanced(Context context, long requestInterval) {
        this.context = context;
        this.requestInterval = requestInterval;
        TTLog.e(TAG, "FusedLocationFetcherBackgroundBalanced: requestInterval = " + requestInterval);
    }


    public boolean isConnected() {

        TTLog.e(TAG, "isConnected");

        if (googleApiClient != null) {
            return googleApiClient.isConnected();
        }
        return false;
    }


    /**
     * Checks if com.example.library.tookan.mylibrary.location fetching is enabled in device or not
     *
     * @param context application context
     * @return true if any com.example.library.tookan.mylibrary.location provider is enabled else false
     */
    public boolean isLocationEnabled(Context context) {

        TTLog.e(TAG, "isLocationEnabled");


        try {
            ContentResolver contentResolver = context.getContentResolver();
            boolean gpsStatus = Settings.Secure.isLocationProviderEnabled(contentResolver, LocationManager.GPS_PROVIDER);
            boolean netStatus = Settings.Secure.isLocationProviderEnabled(contentResolver, LocationManager.NETWORK_PROVIDER);
            return gpsStatus || netStatus;
        } catch (Exception e) {
            e.printStackTrace();
            return false;
        }
    }


    public void connect() {

        TTLog.e(TAG, "connect");

        destroy();
        int resp = GooglePlayServicesUtil.isGooglePlayServicesAvailable(context);
        if (resp == ConnectionResult.SUCCESS) {                                                        // google play services working
            if (isLocationEnabled(context)) {                                                            // com.example.library.tookan.mylibrary.location fetching enabled
                buildGoogleApiClient(context);
            } else {                                                                                    // com.example.library.tookan.mylibrary.location disabled
            }
        } else {                                                                                        // google play services not working
        }
    }


    public void destroy() {

        TTLog.e(TAG, "destroy");

        try {
//            TrackingLog.e("com.example.library.tookan.mylibrary.location", "destroy");
            if (googleApiClient != null) {
                if (googleApiClient.isConnected()) {
                    LocationServices.FusedLocationApi.removeLocationUpdates(googleApiClient, locationIntent);
                    locationIntent.cancel();
                    googleApiClient.disconnect();
                } else if (googleApiClient.isConnecting()) {
                    googleApiClient.disconnect();
                }
            }
        } catch (Exception e) {
            TTLog.e("e", "=" + e.toString());
        }
    }


    protected void createLocationRequest(long interval) {

        TTLog.e(TAG, "createLocationRequest");

        locationrequest = new LocationRequest();
        locationrequest.setInterval(interval);
        locationrequest.setFastestInterval(interval / 2);
        locationrequest.setPriority(LocationRequest.PRIORITY_BALANCED_POWER_ACCURACY);
    }


    protected synchronized void buildGoogleApiClient(Context context) {

        TTLog.e(TAG, "buildGoogleApiClient");

        googleApiClient = new GoogleApiClient.Builder(context)
                .addConnectionCallbacks(this)
                .addOnConnectionFailedListener(this)
                .addApi(LocationServices.API).build();
        googleApiClient.connect();
    }

    protected void startLocationUpdates(long interval) {

        TTLog.e(TAG, "startLocationUpdates");

        createLocationRequest(interval);
        Intent intent = new Intent(context, FusedLocationReceiverBackgroundBalanced.class);
        locationIntent = PendingIntent.getBroadcast(context, LOCATION_PI_ID, intent, PendingIntent.FLAG_UPDATE_CURRENT);
        if (ActivityCompat.checkSelfPermission(context, Manifest.permission.ACCESS_FINE_LOCATION) != PackageManager.PERMISSION_GRANTED
                && ActivityCompat.checkSelfPermission(context, Manifest.permission.ACCESS_COARSE_LOCATION) != PackageManager.PERMISSION_GRANTED) {
            // TODO: Consider calling
            //    ActivityCompat#requestPermissions
            // here to request the missing permissions, and then overriding
            //   public void onRequestPermissionsResult(int requestCode, String[] permissions,
            //                                          int[] grantResults)
            // to handle the case where the user grants the permission. See the documentation
            // for ActivityCompat#requestPermissions for more details.
            return;
        }

        if (googleApiClient != null && googleApiClient.isConnected())
            LocationServices.FusedLocationApi.requestLocationUpdates(googleApiClient, locationrequest, locationIntent);
    }


    @Override
    public void onConnected(Bundle connectionHint) {

        TTLog.e(TAG, "onConnected");

        startLocationUpdates(requestInterval);
    }

    @Override
    public void onConnectionSuspended(int i) {

        TTLog.e(TAG, "onConnectionSuspended");

        destroy();
        connect();
    }

    @Override
    public void onConnectionFailed(@NonNull ConnectionResult result) {

        TTLog.e(TAG, "onConnectionFailed");

        destroy();
        connect();
    }

}

