package tookan.tookanlocationtrackinglibrary.retrofit2;

import android.app.Activity;

import java.lang.ref.WeakReference;
import java.net.ConnectException;
import java.net.SocketTimeoutException;
import java.net.UnknownHostException;

import retrofit2.Call;
import retrofit2.Callback;
import retrofit2.Response;
import tookan.tookanlocationtrackinglibrary.model.BaseModel;

import static tookan.tookanlocationtrackinglibrary.retrofit2.Codes.StatusCode.PARSING_ERROR;


/**
 * Custom Retrofit ResponseResolver
 *
 * @param <T>
 */
public abstract class ResponseResolver<T extends BaseModel> implements Callback<T>
{
	private static final String TAG = ResponseResolver.class.getSimpleName();
	private WeakReference<Activity> weakActivity = null;
	private Boolean showLoading = false;
	private Boolean showError = false;

	private final String NO_INTERNET_MESSAGE = "No internet connection. Please try again later.";
	private final String REMOTE_SERVER_FAILED_MESSAGE = "Application server could not respond. Please try later.";
	public final static String UNEXPECTED_ERROR_OCCURRED = "Something went wrong. Please try again later";


	/**
	 * @param activity
	 */
	public ResponseResolver(Activity activity)
	{
		this.weakActivity = new WeakReference<>(activity);
		this.showError = true;
	}


	/**
	 * @param activity
	 * @param showLoading
	 */
	public ResponseResolver(Activity activity, Boolean showLoading)
	{
		this.weakActivity = new WeakReference<>(activity);
		this.showLoading = showLoading;
	}


	/**
	 * @param activity
	 * @param showLoading
	 * @param showError
	 */
	public ResponseResolver(Activity activity, Boolean showLoading, Boolean showError)
	{
		this.weakActivity = new WeakReference<>(activity);
		this.showLoading = showLoading;
		this.showError = showError;
	}

	public abstract void success(T t);

	public abstract void failure(APIError error);

	@Override
	public void onResponse(Call<T> t, Response<T> tResponse)
	{
		try
		{

			resolve(t, tResponse);

		}
		catch (Throwable e)
		{
			e.printStackTrace();

			// Pipeline the flow of Control
			APIError apiError = new APIError(PARSING_ERROR.getStatusCode(), Constants.StatusDescription.PARSING_ERROR.getMessage(weakActivity.get()));
			fireError(apiError);

		}

	}

	@Override
	public void onFailure(Call<T> t, Throwable throwable)
	{
		fireError(new APIError(900, resolveNetworkError(throwable)));
	}


	private void resolve(Call<T> t, Response<T> tResponse)
	{
		if (tResponse.isSuccessful() && tResponse.body() != null)
		{

			BaseModel baseModel = tResponse.body();

			Codes.StatusCode status = Codes.StatusCode.get(baseModel.getStatus());
			APIError apiError = new APIError(baseModel.getStatus(), baseModel.getMessage());
			switch (status)
			{

				case ACTION_COMPLETE:
					success(tResponse.body());
					break;


				case PARAMETER_MISSING:
				case SHOW_ERROR_MESSAGE:
					fireError(apiError);
					break;

				default:
					fireError(apiError);
					break;
			}


		}
	}


	/**
	 * @param apiError
	 */
	public void fireError(APIError apiError)
	{
		failure(apiError);
	}


	/**
	 * @param apiError
	 * @return
	 */
	public Boolean checkAuthorizationError(APIError apiError)
	{
		return true;
	}

	/**
	 * Method resolve network errors
	 *
	 * @param cause
	 * @return
	 */

	private String resolveNetworkError(Throwable cause)
	{
		if (cause instanceof UnknownHostException)
		{
			return NO_INTERNET_MESSAGE;
		}
		else if (cause instanceof SocketTimeoutException)
		{
			return REMOTE_SERVER_FAILED_MESSAGE;
		}
		else if (cause instanceof ConnectException)
		{
			return REMOTE_SERVER_FAILED_MESSAGE;
		}
		else if(cause instanceof IllegalStateException)
		{
			return "Parse Exception";
		}
		return UNEXPECTED_ERROR_OCCURRED;
	}


}
