package tookan.tookanlocationtrackinglibrary;

import android.Manifest;
import android.content.Context;
import android.content.pm.PackageManager;
import android.content.pm.PackageManager.NameNotFoundException;
import android.os.Build;
import android.support.v4.content.ContextCompat;
import android.telephony.TelephonyManager;

import java.util.Calendar;
import java.util.GregorianCalendar;
import java.util.Locale;
import java.util.TimeZone;

import tookan.tookanlocationtrackinglibrary.metering.Keys;
import tookan.tookanlocationtrackinglibrary.metering.utils.Prefs;

/**
 * Fetches data for which the App is dependent to communicate with Server
 *
 * @author Rishabh
 */
public class Dependencies {

    /*      Specifies the Device Type being used     */
    public static final int ANDROID = 0;

    /*      Specifies the Version Details at Play Store     */
    public static final int STORE_VERSION = 1;
    public static final String STORE_VERSION_NAME = "1.0";



    /**
     * Extracts or generates a device token of the Device
     *
     * @param context
     * @return
     */
    public static String getDeviceToken(Context context) {

        return Prefs.with(context).getString(Keys.Prefs.DEVICE_TOKEN, "");
    }

    /**
     * Retrieves the name of the Device
     *
     * @return
     */
    public static String getDeviceName() {

        return  Build.DEVICE;
    }

    /**
     * Retrieves the name of the Device
     *
     * @return
     */
    public static String getDeviceManufacturer() {

        return android.os.Build.MANUFACTURER;
    }

    /**
     * Retrieves the name of the Device
     *
     * @return
     */
    public static String getDeviceModel() {

        return android.os.Build.MODEL;
    }

    /**
     * Retrieves the OS Version of device
     *
     * @return
     */
    public static String getDeviceOSVersion() {
        return android.os.Build.VERSION.RELEASE;
    }

    /**
     * Retrieves the IMEI Number of the Device
     *
     * @return
     */
    public static String getDeviceId(Context context) {
        if (isPermissionGranted(context, Manifest.permission.READ_PHONE_STATE))
            return ((TelephonyManager) context.getSystemService(Context.TELEPHONY_SERVICE)).getDeviceId();

        return "Permission not granted";
    }


    /**
     * Retrieves the Version of the App
     *
     * @param context
     * @return
     * @throws NameNotFoundException
     */
    public static long getAppVersionCode(Context context) {

        try {
            return context.getPackageManager()
                    .getPackageInfo(context.getPackageName(), 0).versionCode;
        } catch (NameNotFoundException e) {
            e.printStackTrace();
            return STORE_VERSION;
        }
    }

    /**
     * Retrieves the Version of the App
     *
     * @param context
     * @return
     * @throws NameNotFoundException
     */
    public static String getAppVersionName(Context context) {

        try {
            return context.getPackageManager()
                    .getPackageInfo(context.getPackageName(), 0).versionName;
        } catch (NameNotFoundException e) {
            e.printStackTrace();
            return STORE_VERSION_NAME;
        }
    }

    /**
     * Retrieves the code of the Device's Country
     *
     * @param context
     * @return
     */
    public static String getCountryCode(Context context) {

        return context.getApplicationContext().getResources()
                .getConfiguration().locale.getDisplayCountry(Locale
                        .getDefault());
    }

    /**
     * Method to get the TimeZone offset in minutes
     *
     * @return
     */
    public static long getTimeZoneInMinutes() {

        TimeZone tz = TimeZone.getDefault();
        Calendar cal = GregorianCalendar.getInstance(tz);
        long offsetInMillis = tz.getOffset(cal.getTimeInMillis());

        return offsetInMillis / 60000;
    }



    /**
     * Method to save the Device Token
     *
     * @param context
     * @param deviceToken
     */
    public static void saveDeviceToken(Context context, String deviceToken) {

        Prefs.with(context).save(Keys.Prefs.DEVICE_TOKEN, deviceToken);
    }


    public static String getLocale()
    {
       return Locale.getDefault().toString();
    }


    public static boolean isPermissionGranted(Context context, String permission) {
        if(Build.VERSION.SDK_INT < 23) {
            return true;
        } else
        {
            return ContextCompat.checkSelfPermission(context, permission) == PackageManager.PERMISSION_GRANTED;
        }

    }

}
