package app.raybritton.tokenstorage

import app.raybritton.tokenstorage.crypto.Crypto
import app.raybritton.tokenstorage.persistence.Persistence
import io.reactivex.BackpressureStrategy
import io.reactivex.Completable
import io.reactivex.Flowable
import io.reactivex.subjects.ReplaySubject

class RxTokenStorage(val crypto: Crypto,
                     val persistence: Persistence) : RxPersistence {
    private val loadSubjects = mutableMapOf<String, ReplaySubject<Optional<String>>>()

    private val keysSubject = ReplaySubject.create<List<String>>(1)

    override fun clearAll(): Completable {
        return Completable.fromCallable {
            crypto.verify()
            persistence.clearAll()
            loadSubjects.values.forEach {
                it.onNext(Optional.None)
            }
            keysSubject.onNext(listOf())
        }
    }

    override fun clear(key: String): Completable {
        return Completable.fromCallable {
            crypto.verify()
            persistence.clear(key)
            reload(key)
        }
    }

    override fun save(key: String, plaintext: String): Completable {
        return Completable.fromCallable {
            crypto.verify()
            val encrypted = crypto.encrypt(plaintext)
            persistence.save(key, encrypted)
            reload(key)
        }
    }

    override fun load(key: String): Flowable<Optional<String>> {
        crypto.verify()
        if (loadSubjects.containsKey(key)) {
            return loadSubjects[key]!!.share().toFlowable(BackpressureStrategy.LATEST)
        } else {
            val subject = ReplaySubject.create<Optional<String>>(1)
            loadSubjects[key] = subject
            subject.startWith { loadString(key) }
            return subject.share().toFlowable(BackpressureStrategy.LATEST)
        }
    }

    override fun keys(): Flowable<List<String>> {
        crypto.verify()
        return keysSubject.share().toFlowable(BackpressureStrategy.LATEST)
                .startWith(persistence.keys())
    }

    private fun loadString(key: String): String? {
        val encrypted = persistence.load(key)
        if (encrypted == null) {
            return null
        } else {
            return crypto.decrypt(encrypted)
        }
    }
    private fun reload(key: String) {
        loadSubjects[key]?.onNext(Optional.auto(loadString(key)))
        keysSubject.onNext(persistence.keys())
    }

}