package app.raybritton.elog.data

import app.raybritton.elog.ELog
import app.raybritton.elog.ELogConfig
import java.io.File
import java.util.*

internal class LogManager {
    private val db = LogDatabase()

    internal fun getLogs(): List<LogFile> {
        return db.getLogs()
    }

    internal fun getCurrentLog(): LogFile? {
        return db.getLatestLog()
    }

    internal fun setCodeForLog(logFile: LogFile, code: String) {
        val updated = logFile.copy(code = code)
        db.updateLog(updated)
    }

    internal fun createLog(): File {
        val dir = ELog.logDir
        var file = File(dir, UUID.randomUUID().toString())
        while (file.exists()) {
            file = File(dir, UUID.randomUUID().toString())
        }
        if (file.createNewFile()) {
            val logFile = LogFile(
                file.nameWithoutExtension,
                file,
                System.currentTimeMillis(),
                null
            )
            db.insertLog(logFile)
            return file
        } else {
            throw IllegalStateException("Unable to create new log files")
        }
    }

    internal fun deleteLog(id: String) {
        db.deleteLog(id)
        File(ELog.logDir, id).delete()
    }

    internal fun cleanupLogs() {
        val maxUnsubmittedAge = System.currentTimeMillis() - ELogConfig.maxUnsubmittedLogAgeMs
        val maxSubmittedAge = System.currentTimeMillis() - ELogConfig.maxSubmittedLogAgeMs

        val logs = getLogs()

        logs
            .filter { it.code == null }
            .filter { it.created < maxUnsubmittedAge }
            .forEach { deleteLog(it.id) }

        logs
            .filter { it.code != null }
            .filter { it.created < maxSubmittedAge }
            .forEach { deleteLog(it.id) }

        logs
            .filter { it.file.length() == 0L }
            .forEach { deleteLog(it.id) }
    }

    internal fun deleteAll() {
        db.deleteAll()
        ELog.logDir.listFiles()?.forEach { it.delete() }
    }
}