package app.raybritton.elog.data

import android.content.ContentValues
import android.content.Context
import android.database.sqlite.SQLiteDatabase
import android.database.sqlite.SQLiteOpenHelper
import app.raybritton.elog.ELog
import java.io.File

internal class LogDatabase {
    private val openHelper by lazy {
        LogDbOpenHelper(ELog.context, "elog.db", 1)
    }

    internal fun getLogs(): List<LogFile> {
        val cursor = openHelper.readableDatabase.rawQuery("SELECT * FROM logs", arrayOf())
        val results = mutableListOf<LogFile>()

        while (cursor.moveToNext()) {
            results.add(LogFile(
                cursor.getString(cursor.getColumnIndex(COL_ID)),
                File(cursor.getString(cursor.getColumnIndex(COL_FILE))),
                cursor.getLong(cursor.getColumnIndex(COL_TIMESTAMP)),
                cursor.getString(cursor.getColumnIndex(COL_CODE))
            ))
        }

        cursor.close()
        return results
    }

    internal fun insertLog(logFile: LogFile) {
        val values = ContentValues()
        values.put(COL_ID, logFile.id)
        values.put(COL_FILE, logFile.file.absolutePath)
        values.put(COL_TIMESTAMP, logFile.created)
        values.put(COL_CODE, logFile.code)
        openHelper.writableDatabase.insert(TABLE_NAME, null, values)
    }

    internal fun updateLog(logFile: LogFile) {
        val values = ContentValues()
        values.put(COL_ID, logFile.id)
        values.put(COL_FILE, logFile.file.absolutePath)
        values.put(COL_TIMESTAMP, logFile.created)
        values.put(COL_CODE, logFile.code)
        openHelper.writableDatabase.update(TABLE_NAME, values, "$COL_ID = ?", arrayOf(logFile.id))
    }

    internal fun deleteLog(id: String) {
        openHelper.writableDatabase.delete(TABLE_NAME, "$COL_ID = ?", arrayOf(id))
    }

    internal fun deleteAll() {
        openHelper.writableDatabase.execSQL("DELETE FROM $TABLE_NAME")
    }

    fun getLatestLog(): LogFile? {
        val cursor = openHelper.readableDatabase.rawQuery("SELECT * FROM logs LIMIT 1", arrayOf())
        val results = mutableListOf<LogFile>()

        if (cursor.count == 0) {
            return null
        }

        while (cursor.moveToNext()) {
            results.add(LogFile(
                cursor.getString(cursor.getColumnIndex(COL_ID)),
                File(cursor.getString(cursor.getColumnIndex(COL_FILE))),
                cursor.getLong(cursor.getColumnIndex(COL_TIMESTAMP)),
                cursor.getString(cursor.getColumnIndex(COL_CODE))
            ))
        }

        cursor.close()
        return results[0]
    }

    private class LogDbOpenHelper(ctx: Context, name: String, version: Int) : SQLiteOpenHelper(ctx, name, null, version) {
        override fun onCreate(db: SQLiteDatabase) {
            db.execSQL("CREATE TABLE $TABLE_NAME ($COL_ID TEXT PRIMARY KEY, $COL_FILE TEXT, $COL_CODE TEXT, $COL_TIMESTAMP INTEGER)")
        }

        override fun onUpgrade(db: SQLiteDatabase, oldVer: Int, newVer: Int) {

        }
    }

    private companion object {
        private const val TABLE_NAME = "logs"
        private const val COL_ID = "id"
        private const val COL_FILE = "file"
        private const val COL_CODE = "code"
        private const val COL_TIMESTAMP = "dateFormatter"
    }
}