package app.raybritton.elog.ui

import android.app.Activity
import android.app.AlertDialog
import android.content.ClipData
import android.content.ClipboardManager
import android.content.Context
import android.content.Intent
import android.os.Bundle
import android.view.View
import android.widget.TextView
import android.widget.Toast
import androidx.core.content.FileProvider
import app.raybritton.elog.BuildConfig
import app.raybritton.elog.ELog
import app.raybritton.elog.ELog.context
import app.raybritton.elog.ELogConfig
import app.raybritton.elog.R
import app.raybritton.elog.arch.LogModule
import app.raybritton.elog.data.LogFile
import app.raybritton.elog.data.Result
import kotlinx.android.synthetic.main.elog_activity_submit.*
import java.text.SimpleDateFormat
import java.util.*
import kotlin.concurrent.thread


internal class LogSubmitActivity : Activity() {
    private val dateFormatter = SimpleDateFormat("yyyy/MM/dd HH:mm:ss", Locale.UK)

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        setContentView(R.layout.elog_activity_submit)
        setStatusBarColor(ELogConfig.toolbarBackgroundColor)

        val logFile = intent.getSerializableExtra(ARG_LOG_FILE) as LogFile

        thread {
            LogModule.logUploader.upload(logFile.file) { result ->
                runOnUiThread {
                    elog_submit_progress.visibility = View.GONE
                    when (result) {
                        is Result.Success -> {
                            elog_submit_text.text = getString(R.string.elog_send_success, ELog.contactName)
                            if (result.code.all { it.isDigit() }) {
                                setCopyOnClick(elog_submit_code_number, result.code)
                            } else {
                                setCopyOnClick(elog_submit_code_mixed, result.code)
                            }
                            setResult(RESULT_OK)
                            elog_submit_text.visibility = View.VISIBLE
                            elog_submit_close.visibility = View.VISIBLE
                            elog_submit_close.setOnClickListener {
                                finish()
                            }
                            thread {
                                LogModule.logManager.setCodeForLog(logFile, result.code)
                            }
                        }
                        is Result.Error -> {
                            AlertDialog.Builder(this)
                                .setMessage(getString(R.string.elog_send_failed_message, result.e.message))
                                .setNegativeButton(R.string.elog_generic_cancel) { _, _ -> finish() }
                                .setPositiveButton(R.string.elog_send_failed_button_email) { _, _ ->
                                    sendEmail(logFile)
                                }
                                .show()
                        }
                    }
                }
            }
        }
    }

    private fun setCopyOnClick(tv: TextView, code: String) {
        tv.text = code
        tv.visibility = View.VISIBLE
        tv.setOnClickListener {
            val clipboard = getSystemService(Context.CLIPBOARD_SERVICE) as ClipboardManager
            val clip = ClipData.newPlainText("code", tv.text)
            clipboard.setPrimaryClip(clip)
            Toast.makeText(this, getString(R.string.elog_copied), Toast.LENGTH_SHORT).show()
        }
    }

    private fun sendEmail(logFile: LogFile) {
        val intent = Intent(Intent.ACTION_SEND)
        intent.putExtra(Intent.EXTRA_EMAIL, arrayOf(ELog.contactEmail))
        intent.putExtra(Intent.EXTRA_SUBJECT, BuildConfig.APPLICATION_ID + " crash")
        intent.type = "text/plain"
        intent.addFlags(Intent.FLAG_GRANT_READ_URI_PERMISSION)
        val uri = FileProvider.getUriForFile(context, "app.raybritton.elog.FileProvider", logFile.file)
        intent.putExtra(Intent.EXTRA_STREAM, uri)
        intent.putExtra(Intent.EXTRA_TEXT, "Crash occurred at " + dateFormatter.format(Date(logFile.created)))
        intent.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK)
        startActivity(Intent.createChooser(intent, getString(app.raybritton.elog.R.string.elog_action_send)))
        finish()
    }

    companion object {
        private const val ARG_LOG_FILE = "logfile.obj"

        internal fun start(ctx: Activity, logFile: LogFile) {
            val intent = Intent(ctx, LogSubmitActivity::class.java)
            intent.putExtra(ARG_LOG_FILE, logFile)
            ctx.startActivity(intent)
        }
    }
}