package app.raybritton.tokenstorage.room

import android.content.Context
import androidx.room.Room
import app.raybritton.tokenstorage.persistence.Persistence

/**
 * Uses the Room library to persist strings to SQLite
 *
 * @param filename The name of the database file
 *
 * @param inMemory If true the database will be created in memory
 */
class RoomPersistence(private val context: Context,
                      private val filename: String = "tokens.db",
                      private val inMemory: Boolean = false) : Persistence {

    private val db by lazy {
        if (inMemory) {
            Room.inMemoryDatabaseBuilder(context, CryptoDatabase::class.java)
                    .allowMainThreadQueries()
                    .build()
        } else {
            Room.databaseBuilder(context, CryptoDatabase::class.java, filename).build()
        }
    }

    override fun clearAll() {
        db.cryptoDao().deleteAll()
    }

    override fun clear(key: String) {
        db.cryptoDao().delete(key)
    }

    override fun save(key: String, encrypted: String) {
        db.cryptoDao().insert(Encrypted(key, encrypted))
    }

    override fun load(key: String): String? {
        return db.cryptoDao().get(key)?.value
    }

    override fun keys(): List<String> {
        return db.cryptoDao().getKeys()
    }

    override fun contains(key: String): Boolean {
        return db.cryptoDao().get(key) != null
    }

    override fun reset() {
        db.clearAllTables()
        db.close()
    }

    fun close() {
        db.close()
    }
}