package app.raybritton.tokenstorage.crypto

import app.raybritton.tokenstorage.CryptoLogging
import java.io.File
import java.util.UUID

/**
 * Uses a passphrase (string) to encrypt and decrypt the strings, for example this could be
 * a PIN entered by the user. (AES/CBC/PKCS5Padding)
 *
 * Important:
 * passphrase MUST be set before calling any method
 *
 * A salt is generated when using this class, it is saved a file called 'crypto-salt' (by default) in
 * the apps files dir, if this is deleted or changed the stored data will be lost
 */
class PassphraseCrypto(private val saltDir: File,
                       private val saltFilename: String = "crypto-salt") : Crypto, CryptoLogging {
    private var secretKeys: AesCbcWithIntegrity.SecretKeys? = null

    init {
        fine("init")
    }

    /**
     * This can be set at any point, any number of times
     * When changed the crypto is rebuilt, if the passphrase does not match the
     * passphrase used to encrypt the data it will be mangled when decrypting but not lost
     *
     */
    var passphrase: String? = null
        @Synchronized set(value) {
            debug("passphrase set to ${value.shrink()}")
            debug("secretKeys already at ${secretKeys.shrink()}")
            if (field == value) {
                debug("matches passphrase")
                return
            }
            field = value
            invalidate()
        }

    private val salt by lazy {
        fine("salt creation")
        val saltFile = File(saltDir, saltFilename)
        if (saltFile.exists()) {
            debug("salt exists")
            saltFile.readLines()[0]
        } else {
            val newSalt = UUID.randomUUID().toString()
            debug("created salt: $newSalt")
            saltFile.writeText(newSalt)
            newSalt
        }
    }

    private fun invalidate() {
        fine("invalidate()")
        if (passphrase == null) {
            debug("passphrase null, clearing secretKeys")
            secretKeys = null
        } else {
            debug("creating secretKeys")
            secretKeys = AesCbcWithIntegrity.generateKeyFromPassword(passphrase!!, salt)
            debug("set to $secretKeys")
        }
    }

    override fun encrypt(plaintext: String): String {
        fine("encrypt(${plaintext.shrink()})")
        return AesCbcWithIntegrity.encrypt(plaintext, secretKeys!!).toString()
    }

    override fun decrypt(encrypted: String): String {
        fine("decrypt(${encrypted.shrink()})")
        val mac = AesCbcWithIntegrity.CipherTextIvMac(encrypted)
        return String(AesCbcWithIntegrity.decrypt(mac, secretKeys!!))
    }

    override fun reset() {
        fine("reset()")
        File(saltDir, saltFilename).delete()
    }

    override fun verify() {
        fine("verify()")
        if (passphrase == null || secretKeys == null) {
            debug("secretKeys: ${secretKeys.shrink()}, passphrase: ${passphrase.shrink()}")
            throw IllegalStateException("passphrase must be set before use")
        }
        if (passphrase.isNullOrEmpty()) {
            throw IllegalStateException("passphrase must have content")
        }
    }
}