package app.pitech.event.worker

import android.arch.persistence.room.Room
import android.content.Context
import androidx.work.Worker
import androidx.work.WorkerParameters
import app.pitech.event.Pipeline
import app.pitech.event.db.PayloadDatabase
import app.pitech.event.getTimeInMillis
import app.pitech.event.isNetwork
import kotlin.concurrent.thread

/**
 * Workmanager that fires the payload events that are stored in the database.
 * Post firing the event, the said event is deleted from the db
 */
internal class FireStoredEventsWork(val context: Context, workerParams: WorkerParameters) :
    Worker(context, workerParams) {

    private val eventDb by lazy {
        Room.databaseBuilder(
            context,
            PayloadDatabase::class.java, "pipeline-db"
        )
            .fallbackToDestructiveMigration()
            .build()
    }

    /**
     * If the device is connected to the network,
     * @see isNetwork fetch all the offline entries from the db and try to retry them.
     * For every successful request, that event is deleted from the database
     */
    override fun doWork(): Result {
        return if (isNetwork(context)) {
            eventDb.eventDao().query().forEach { message ->
                message.sentAt = getTimeInMillis()
                Pipeline.with(context).pushEvent(message) {
                    thread {
                        if (it) eventDb.eventDao().delete(message)
                    }
                }
            }
            Result.success()
        } else
            Result.retry()
    }
}