package app.pitech.event

import android.app.Activity
import android.app.Application
import android.arch.persistence.room.Room
import android.content.Context
import android.content.Context.MODE_PRIVATE
import android.content.SharedPreferences
import android.content.pm.PackageManager
import android.content.res.Configuration
import android.os.Build
import android.os.Bundle
import android.util.Log
import android.webkit.WebSettings
import androidx.work.*
import app.pitech.event.db.PayloadDatabase
import app.pitech.event.model.Address
import app.pitech.event.model.Extra
import app.pitech.event.model.Payload
import app.pitech.event.model.Traits
import app.pitech.event.network.ClientGenerator
import app.pitech.event.network.PipelineClient
import app.pitech.event.worker.FireStoredEventsWork
import okhttp3.ResponseBody
import retrofit2.Call
import retrofit2.Callback
import retrofit2.Response
import java.util.*
import java.util.concurrent.TimeUnit
import kotlin.concurrent.thread

/**
 * Pipeline that pushes the fired events to PubSub
 */
class Pipeline private constructor(context: Context) {

    private val ctx by lazy {
        context
    }

    /**
     * Payload that is initialized with an empty value for all its fields
     */
    private val payload: Payload = Payload()

    /**
     * API key will be stored as a companion object
     */
    private var key: String? = null

    /**
     * Event database is initialized and stored once
     */
    private var eventDb: PayloadDatabase? = null

    /**
     * Traits containing user details that are stored for future usage
     */
    private val traits: Traits = Traits()

    /**
     * Constraints that outline when the work should be fired
     */
    private val constraints = Constraints.Builder().setRequiredNetworkType(NetworkType.CONNECTED).build()

    /**
     * A worker object that handles sending offline events when the device is connected to the network
     */
    private val fireStoredEventsWork = OneTimeWorkRequestBuilder<FireStoredEventsWork>()
        .setConstraints(constraints)
        .setInitialDelay(1, TimeUnit.MINUTES)
        .build()

    /**
     * Content type to be sent by the request
     */
    private val contentType = "application/json"

    /**
     * Retrofit service that will make the network calls
     */
    private lateinit var pipelineService: PipelineClient

    /**
     *  Returns the device resolution with the form width*height, TODO : improve this
     */
    private val deviceScreenResolution by lazy {
        getDisplaySize(ctx)
    }

    /**
     * Returns the installer ID for the app
     */
    private val installerId by lazy {
        packageManager.getInstallerPackageName(applicationId) ?: "Unknown"
    }

    /**
     * Device language, can be English, Hindi, etc.
     */
    private val deviceLanguage by lazy {
        Locale.getDefault().displayLanguage
    }

    /**
     * Return the name with the app by using the context provided by the content provider
     */
    private val applicationName by lazy {
        packageManager
            .getApplicationLabel(
                packageManager
                    .getApplicationInfo(
                        applicationId,
                        PackageManager.GET_META_DATA
                    )
            ).toString()
    }

    /**
     * Returns the device user agent or the http.agent for android version below Jelly Bean
     * The UA is with the following format :
     * Mozilla/5.0 (Linux; Android 5.1; Custom Build/LMY47D) AppleWebKit/537.36 (KHTML, like Gecko) Version/4.0 Chrome/39.0.0.0 Mobile Safari/537.36
     */
    private val deviceUserAgent by lazy {
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN_MR1) {
            WebSettings.getDefaultUserAgent(ctx)
        } else System.getProperty("http.agent")!!
    }

    /**
     * Returns if the device is a tablet or not
     * @return true if the device on which the app is running is a tablet, false otherwise
     */
    private fun isTablet(context: Context): Boolean {
        return context.resources.configuration.screenLayout and Configuration.SCREENLAYOUT_SIZE_MASK >= Configuration.SCREENLAYOUT_SIZE_LARGE
    }

    /**
     * Returns the display size with the phone on which the library is being used
     * @return Display size with the format width * height, TODO : improve the logic
     */
    private fun getDisplaySize(context: Context): String {
        val configuration = context.resources.configuration
        val screenWidthDp = configuration.screenWidthDp
        val screenHeightDp = configuration.screenHeightDp

        return "$screenWidthDp x $screenHeightDp"
    }

    /**
     * Channel with the device Tablet for a tablet, Mobile Phone for a mobile device
     */
    private val deviceChannel by lazy {
        if (isTablet(ctx)) "Tablet" else "Mobile Phone"
    }

    /**
     * Package name with the app
     */
    private val applicationId by lazy {
        ctx.packageName
    }

    private val packageManager by lazy {
        ctx.packageManager
    }

    private val pref: SharedPreferences by lazy {
        ctx.getSharedPreferences("pipeline", MODE_PRIVATE)
    }

    /**
     * Callback that's invoked every time a new activity's lifecycle method was called
     */
    private val activityLifecycleCallbacks = object : Application.ActivityLifecycleCallbacks {

        /**
         * Called whenever an activity's onCreate method was called, automatically sends a pageView call to the server
         */
        override fun onActivityCreated(activity: Activity?, savedInstanceState: Bundle?) {

            /*
             Every time a new activity is started, push the current screen name
              */
            payload.context.screenName = activity?.javaClass?.simpleName.toString()
            push("${activity?.javaClass?.simpleName} Started")
        }

        override fun onActivityStarted(activity: Activity?) = Unit

        override fun onActivityResumed(activity: Activity?) {
            /*
            To prevent activity leaks, set the screenName everytime an activity is resumed.
            Setting the context globally resulted in a memory leak with the activity variable
             */
            payload.context.screenName = activity?.javaClass?.simpleName.toString()
        }

        override fun onActivityPaused(activity: Activity?) {
            payload.context.screenName = ""
        }

        override fun onActivitySaveInstanceState(activity: Activity?, outState: Bundle?) = Unit

        /**
         * When an activity was stopped, fire a worker that retries the requests stored in the database
         */
        override fun onActivityStopped(activity: Activity?) {
            WorkManager
                .getInstance()
                .enqueueUniqueWork(
                    "fire-stored-events",
                    ExistingWorkPolicy.REPLACE,
                    fireStoredEventsWork
                )
        }

        override fun onActivityDestroyed(activity: Activity?) = Unit

    }

    /**
     * This method Initializes the library by :
     * 1. Storing the API_KEY to the local variable key
     * 2. Initializing the pipelineService which will be used for networking purposes
     * 2. Registers the activityLifecycleCallbacks listener so that the library receives callbacks for each activity's lifecycle method
     * 3. Creates the database and saves it to the variable eventDb
     * 4. Calls @link sendVerify with the API key to validate the API KEY
     *
     * @param apiKey : API KEY to be provided by the developer
     * @param url : BASE_URL for the appengine endpoint provided to the developer by solution360
     *
     * As a developer you don't need to call this method directly, the Library uses a content provider
     * @see app.pitech.event.provider.PipelineProvider
     * which calls this method and initializes the api key and other variables for you.
     */
    fun start(apiKey: String, url: String) {
        pipelineService = ClientGenerator.createService(PipelineClient::class.java, url)

        key = apiKey
        (ctx.applicationContext as Application).registerActivityLifecycleCallbacks(activityLifecycleCallbacks)
        eventDb = Room.databaseBuilder(
            ctx,
            PayloadDatabase::class.java, "pipeline-db"
        )
            .fallbackToDestructiveMigration()
            .build()

        sendVerify()
    }

    /**
     * If the key with the name anonymous_id already exists in the preferences,
     * read it from there else Generates and store a UUID to the shared preferences.
     *
     * @return a random UUID that is the same for a user across all the sessions unless the data with the app was deleted
     */
    private fun getAnonymousID(): String {
        var uuid: String = pref.getString("anonymous_id", "")

        return if (uuid.isNotEmpty()) {
            uuid
        } else {
            uuid = getRandomUUID()
            pref.edit().putString("anonymous_id", uuid).apply()
            uuid
        }
    }

    /**
     * This method is called every time before a request is called or
     * @see pushEvent() is called.
     * This method generates a new Payload object and fills it with meta-info that's to be auto-detected
     *
     * @return a payload object which contains all the values that could have been generated automatically by the library
     */
    private fun generateMeta(): Payload {
        payload.channel = deviceChannel
        with(payload.context) {
            appId = applicationId
            language = deviceLanguage
            screenResolution = deviceScreenResolution
            operatingSystem = "Android"
            operatingSystemVersion = Build.VERSION.RELEASE
            mobileDeviceBranding = Build.BRAND
            mobileDeviceModel = Build.MODEL
            mobileDeviceMarketingName = Build.PRODUCT
            appInstallerId = installerId
            userAgent = deviceUserAgent
            appName = applicationName
            appVersion = BuildConfig.VERSION_NAME
        }
        payload.messageId = getRandomUUID()
        payload.timestamp = getTimeInMillis()
        payload.traits = traits
        payload.version = BuildConfig.VERSION_NAME
        payload.anonymousId = getAnonymousID()
        payload.apiKey = key.toString()
        return payload
    }

    /**
     * Calls pushEvent and tracks a user action
     * @param eventName Name with the event, this can be button click, scroll, etc.
     * @param extras An object with the type
     * @see Extra
     * that contains any extra dimension that is to be sent with the request
     *
     */
    @JvmOverloads
    fun push(eventName: String, extras: Extra = Extra()) {
        if (eventName.isNotEmpty()) {
            val message = generateMeta()
            message.event.name = eventName
            message.extra = extras
            message.traits = traits
            pushEvent(message) {}
        } else
            Log.e(TAG, "Event name can't be empty")
    }

    /**
     * Sends a request to the /verify endpoint to check if the API_KEY being used is valid.
     * In case the api key is invalid, an error is posed to the logcat
     */
    private fun sendVerify() {
        key?.let {
            pipelineService.verify(contentType, it)
                .enqueue(object : Callback<ResponseBody> {
                    override fun onFailure(call: Call<ResponseBody>, t: Throwable) = Unit

                    override fun onResponse(call: Call<ResponseBody>, response: Response<ResponseBody>) {
                        if (!response.isSuccessful) {
                            Log.e(TAG, "Invalid API key, please ensure that you are using the correct API key")
                        }
                    }
                })
        } ?: Log.e(
            TAG,
            "config.json was not found, either ensure that config.json was properly placed in the /res/raw folder or start the library manually by calling Pipeline.with(context).start(key,baseUrl)"
        )
    }

    /**
     * This function is invoked by
     * @see push
     * It tries to send a request to the /publish endpoint and in case the request failed, it saves the request to the
     * database so that it can be retried at a later point with time.
     */
    @JvmSynthetic
    internal fun pushEvent(message: Payload, result: (Boolean) -> Unit) {
        key?.let {
            if (isNetwork(ctx)) {
                message.sentAt = getTimeInMillis()
                pipelineService.publish(contentType, message)
                    .enqueue(object : Callback<ResponseBody> {
                        override fun onFailure(call: Call<ResponseBody>, t: Throwable) {
                            t.printStackTrace()
                            saveEventInDb(message)
                        }

                        override fun onResponse(call: Call<ResponseBody>, response: Response<ResponseBody>) {
                            result(true)
                        }
                    })
            } else {
                saveEventInDb(message)
            }
        } ?: Log.e(
            TAG,
            "config.json was not found, either ensure that config.json was properly placed in the /res/raw folder or start the library manually by calling Pipeline.with(context).start(key,baseUrl)"
        )
    }

    /**
     * Saves the
     * @param message
     * payload to the database so that it can be retried later
     */
    private fun saveEventInDb(message: Payload) {
        thread {
            eventDb?.eventDao()?.insert(message)
        }
    }

    /**
     * This method clears any information stored in the
     * @see traits variable
     */
    fun removeUser() {
        with(traits) {
            userName = ""
            userEmail = ""
            userId = ""
            userAddress = Address("", "", "")
        }
    }

    /**
     * TODO : Improve this method and create a clearUser method
     * Set user details like
     * @param name,
     * @param email,
     * @param userId and
     * @param address as a global variable
     */
    @JvmOverloads
    fun defineUser(
        name: String = "",
        email: String,
        id: String = "",
        city: String = "",
        state: String = "",
        country: String = ""
    ) {
        with(traits) {
            userName = name
            userEmail = email
            userId = id
            val address = Address(city, state, country)
            userAddress = address
        }
    }

    /**
     * Common params that wil be constant across all instances with the Pipeline object
     */
    companion object {

        private var pipeline: Pipeline? = null

        /**
         * Returns an existing instance with the Pipeline class if it exists, else returns a new instance
         */
        @JvmStatic
        fun with(context: Context): Pipeline {
            if (pipeline == null) {
                pipeline = Pipeline(context)
            }
            return pipeline!!
        }

        /**
         * Logcat tag for debugging purpose
         */
        private val TAG = javaClass::class.java.simpleName
    }
}