package app.pitech.event

import android.app.Activity
import android.app.Application
import android.arch.persistence.room.Room
import android.content.Context
import android.content.Context.MODE_PRIVATE
import android.content.SharedPreferences
import android.content.pm.PackageManager
import android.content.res.Configuration
import android.net.ConnectivityManager
import android.os.Build
import android.os.Bundle
import android.util.Log
import android.webkit.WebSettings
import androidx.work.*
import app.pitech.event.db.PayloadDatabase
import app.pitech.event.model.Address
import app.pitech.event.model.Payload
import app.pitech.event.model.Extra
import app.pitech.event.model.Traits
import app.pitech.event.network.ClientGenerator
import app.pitech.event.network.Event360Client
import app.pitech.event.worker.FireStoredEventsWork
import okhttp3.ResponseBody
import retrofit2.Call
import retrofit2.Callback
import retrofit2.Response
import java.util.*
import java.util.concurrent.TimeUnit
import kotlin.concurrent.thread

class Event360 private constructor(private var ctx: Context) {

    /**
     * Common params that wil be constant across all instances of the Event360 object
     */
    companion object {
        /**
         * static method to get an instance of the Event360 class
         */
        fun with(context: Context) = Event360(context)

        /**
         * API key will be stored as a companion object
         */
        internal var key: String? = null

        /**
         * Event database is initialized and stored once
         */
        internal var eventDb: PayloadDatabase? = null

        /**
         * Logcat tag for debugging purpose
         */
        internal const val TAG = "Event360"

        /**
         * Traits containing user details are stored inside the companion object once for future usage
         */
        private var traits: Traits = Traits(Address())

        /**
         * Constraints that outline when the work should be fired
         */
        private val constraints = Constraints.Builder().setRequiredNetworkType(NetworkType.CONNECTED).build()

        /**
         * A worker object that handles sending offline events when the device is connected to the network
         */
        private val fireStoredEventsWork = OneTimeWorkRequestBuilder<FireStoredEventsWork>()
            .setConstraints(constraints)
            .setInitialDelay(1, TimeUnit.HOURS)
            .build()

        private const val contentType = "application/json"

        /**
         * Retrofit service that will make the network calls, TODO : allow this to also take the BASE_URL
         */
        private lateinit var event360Service: Event360Client
    }

    /**
     *  Returns the device resolution of the form width*height, TODO : improve this
     */
    private val deviceScreenResolution by lazy {
        getDisplaySize(ctx)
    }

    /**
     * Returns the installer ID for the app
     */
    private val installerId by lazy {
        packageManager.getInstallerPackageName(applicationId) ?: "Unknown"
    }

    private val deviceLanguage by lazy {
        Locale.getDefault().displayLanguage
    }

    /**
     * Return the name of the app by using the context provided by the content provider
     */
    private val applicationName by lazy {
        packageManager
            .getApplicationLabel(
                packageManager
                    .getApplicationInfo(
                        applicationId,
                        PackageManager.GET_META_DATA
                    )
            ).toString()
    }

    /**
     * Returns the device user agent or the http.agent for android version below Jelly Bean
     * The UA is of the following format :
     * Mozilla/5.0 (Linux; Android 5.1; Custom Build/LMY47D) AppleWebKit/537.36 (KHTML, like Gecko) Version/4.0 Chrome/39.0.0.0 Mobile Safari/537.36
     */
    private val deviceUserAgent by lazy {
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN_MR1) {
            WebSettings.getDefaultUserAgent(ctx)
        } else System.getProperty("http.agent")!!
    }

    /**
     * Returns if the device is a tablet or not
     * @return true if the device on which the app is running is a tablet, false otherwise
     */

    private fun isTablet(context: Context): Boolean {
        return context.resources.configuration.screenLayout and Configuration.SCREENLAYOUT_SIZE_MASK >= Configuration.SCREENLAYOUT_SIZE_LARGE
    }

    /**
     * Returns the display size of the phone on which the library is being used
     * @return Display size of the format width * height, TODO : improve the logic
     */

    private fun getDisplaySize(context: Context): String {
        val configuration = context.resources.configuration
        val screenWidthDp = configuration.screenWidthDp
        val screenHeightDp = configuration.screenHeightDp

        return "$screenWidthDp x $screenHeightDp"
    }

    /**
     * Channel of the device Tablet for a tablet, Mobile Phone for a mobile device
     */
    private val deviceChannel by lazy {
        if (isTablet(ctx)) "Tablet" else "Mobile Phone"
    }

    /**
     * Package name of the app
     */

    private val applicationId by lazy {
        ctx.packageName
    }

    private val packageManager by lazy {
        ctx.packageManager
    }

    private val pref: SharedPreferences by lazy {
        ctx.getSharedPreferences("event360", MODE_PRIVATE)
    }

    /**
     * Callback that's invoked every time a new activity's lifecycle method was called
     */
    private val activityLifecycleCallbacks = object : Application.ActivityLifecycleCallbacks {
        override fun onActivityPaused(activity: Activity?) {
        }

        override fun onActivityResumed(activity: Activity?) {
        }

        override fun onActivityStarted(activity: Activity?) {
        }

        override fun onActivityDestroyed(activity: Activity?) {
        }

        override fun onActivitySaveInstanceState(activity: Activity?, outState: Bundle?) {
        }

        /**
         * When an activity was stopped, fire a worker that retries the requests stored in the database
         */
        override fun onActivityStopped(activity: Activity?) {
            WorkManager
                .getInstance()
                .enqueueUniqueWork(
                    "fire-stored-events",
                    ExistingWorkPolicy.REPLACE,
                    fireStoredEventsWork
                )
        }

        /**
         * Called whenever an activity's onCreate method was called, automatically sends a pageview call to the server
         */
        override fun onActivityCreated(activity: Activity?, savedInstanceState: Bundle?) {
            activity?.let { ctx = it }
            track("${ctx.javaClass.simpleName} Started")
        }
    }

    /**
     * This method Initializes the library by :
     * 1. Storing the API_KEY to the companion object key
     * 2. Initializing the event360Service variable in companion object
     * 2. Registers the activityLifecycleCallbacks listener so that the library receives callbacks for each activity's lifecycle method
     * 3. Creates the database and saves it to the companion object eventDb
     * 4. Calls @link sendVerify with the API key to validate the API KEY
     *
     * @param apiKey : API KEY to be provided by the developer
     * @param url : BASE_URL for the appengine endpoint provided to the developer by solution360
     *
     * As a developer you don't need to call this method directly, the Library uses a content provider @link EventProvider
     * which calls this method and initializes the api key and other variables for you.
     *
     */
    fun initialize(apiKey: String, url: String) {
        event360Service = ClientGenerator.createService(Event360Client::class.java, url)

        key = apiKey
        (ctx.applicationContext as Application).registerActivityLifecycleCallbacks(activityLifecycleCallbacks)
        eventDb = Room.databaseBuilder(
            ctx,
            PayloadDatabase::class.java, "event360-db"
        ).build()

        sendVerify()
    }

    /**
     * Sends a request to the /verify endpoint to check if the API_KEY being used is valid.
     * In case the api key is invalid, an error is posed to the logcat
     */
    private fun sendVerify() {
        key?.let {
            event360Service.verify(contentType, it)
                .enqueue(object : Callback<ResponseBody> {
                    override fun onFailure(call: Call<ResponseBody>, t: Throwable) = Unit

                    override fun onResponse(call: Call<ResponseBody>, response: Response<ResponseBody>) {
                        if (response.code() == 403 || response.code() == 502) {
                            Log.e(TAG, "Invalid API key, please ensure that you are using the correct API key")
                        }
                    }
                })
        } ?: Log.e(
            TAG,
            "config.json was not found, either ensure that config.json was properly placed in the /res/raw folder or initialize the library manually by calling Event360.with(context).initialize(key,baseUrl)"
        )
    }

    /**
     * If the key with the name anonymous_id already exists in the preferences,
     * read it from there else Generates and store a UUID to the shared preferences.
     *
     * @return a random UUID that is the same for a user across all the sessions unless the data of the app was deleted
     */
    private fun getAnonymousID(): String {
        val uuid: String
        return if (pref.getString("anonymous_id", "").isEmpty()) {
            uuid = UUID.randomUUID().toString()
            pref.edit().putString("anonymous_id", uuid).apply()
            uuid
        } else {
            pref.getString("anonymous_id", "")
        }
    }

    /**
     * This method is called everytime before a request is called or
     * @see sendEvent() is called.
     * This method generates a new Payload object and fills it with meta-info that's to be auto-detected
     *
     * @return a payload object which contains all the values that could have been generated automatically by the library
     */
    private fun generateMeta(): Payload {
        val message = Payload()
        message.channel = deviceChannel
        with(message.context) {
            appId = applicationId
            language = deviceLanguage
            screenResolution = deviceScreenResolution
            operatingSystem = "Android"
            operatingSystemVersion = Build.VERSION.RELEASE
            mobileDeviceBranding = Build.BRAND
            mobileDeviceModel = Build.MODEL
            mobileDeviceMarketingName = Build.PRODUCT
            appInstallerId = installerId
            userAgent = deviceUserAgent
            appName = applicationName
            appVersion = BuildConfig.VERSION_NAME
            screenName = ctx.javaClass.simpleName
        }
        message.messageId = UUID.randomUUID().toString()
        message.timestamp = TimeUnit.MILLISECONDS.toSeconds(System.currentTimeMillis()).toString()
        message.traits = traits
        message.version = BuildConfig.VERSION_NAME
        message.anonymousId = getAnonymousID()
        message.apiKey = key.toString()
        return message
    }

    /**
     * @see track(eventName,extras)
     */

    fun track(eventName: String) {
        track(eventName, Extra())
    }

    /**
     * Calls sendEvent and tracks a user action
     * @param eventName Name of the event, this can be button click, scroll, etc.
     * @param extras An object of the type
     * @see Extra
     * that contains any extra dimension that is to be sent with the request
     *
     */
    fun track(eventName: String, extras: Extra) {
        val message = generateMeta()
        message.event.name = eventName
        message.extra = extras
        message.traits = traits
        sendEvent(message) {}
    }

    /**
     * This function is invoked by
     * @see track
     * It tries to send a request to the /publish endpoint and in case the request failed, it saves the request to the
     * database so that it can be retried at a later point of time.
     */
    internal fun sendEvent(message: Payload, result: (Boolean) -> Unit) {
        key?.let {
            if (isNetwork(ctx)) {
                message.sentAt = TimeUnit.MILLISECONDS.toSeconds(System.currentTimeMillis()).toString()
                event360Service.publish(contentType, message)
                    .enqueue(object : Callback<ResponseBody> {
                        override fun onFailure(call: Call<ResponseBody>, t: Throwable) {
                            t.printStackTrace()
                            saveEventInDb(message)
                        }

                        override fun onResponse(call: Call<ResponseBody>, response: Response<ResponseBody>) {
                            result(true)
                        }
                    })
            } else {
                saveEventInDb(message)
            }
        } ?: Log.e(
            TAG,
            "config.json was not found, either ensure that config.json was properly placed in the /res/raw folder or initialize the library manually by calling Event360.with(context).initialize(key,baseUrl)"
        )
    }

    /**
     * Saves the
     * @param payload
     * message to the database so that it can be retried later
     */
    private fun saveEventInDb(message: Payload) {
        thread {
            eventDb?.eventDao()?.insert(message)
        }
    }

    /**
     * Set user details like
     * @param name,
     * @param email,
     * @param userId and
     * @param address to the companion object named
     * @see traits so that it can be sent automatically for every request when
     * @see generateMeta was called
     */
    fun setUser(name: String, email: String, userId: String, address: Address) {
        traits = Traits()
        with(traits) {
            this.name = name
            this.email = email
            this.userId = userId
            this.address = address
        }
    }
}

/**
 * Checks to see if the device is connected to the network
 * @return true if the device is connected to a network else false
 * @param context Application or activity context
 */
internal fun isNetwork(context: Context): Boolean {
    val connectivityManager = context.getSystemService(Context.CONNECTIVITY_SERVICE) as ConnectivityManager
    val activeNetworkInfo = connectivityManager.activeNetworkInfo
    return activeNetworkInfo != null && activeNetworkInfo.isConnected
}