package app.pitech.event

import android.app.Activity
import android.app.Application
import android.arch.persistence.room.Room
import android.content.Context
import android.content.Context.MODE_PRIVATE
import android.content.SharedPreferences
import android.content.pm.PackageManager
import android.content.res.Configuration
import android.net.ConnectivityManager
import android.os.Build
import android.os.Bundle
import android.util.Log
import android.webkit.WebSettings
import androidx.work.*
import app.pitech.event.db.EventDatabase
import app.pitech.event.model.Address
import app.pitech.event.model.Payload
import app.pitech.event.model.Extra
import app.pitech.event.model.Traits
import app.pitech.event.network.ClientGenerator
import app.pitech.event.network.Event360Client
import app.pitech.event.worker.FireStoredEventsWork
import okhttp3.ResponseBody
import retrofit2.Call
import retrofit2.Callback
import retrofit2.Response
import java.util.*
import java.util.concurrent.TimeUnit
import kotlin.concurrent.thread

class Analytics private constructor(private var ctx: Context) {

    // Common params that wil be constant across all instances of the Analytics object
    companion object {
        //static method to get an instance of the Analytis class
        fun of(context: Context) = Analytics(context)

        //API key will be stored as a companion object
        internal var key: String? = null
        //Event database is initialized and stored once
        internal var eventDb: EventDatabase? = null

        //Logcat tag for debugging purpose
        internal const val TAG = "Analytics"

        //Traits containing user details are stored inside the companion object once for future usage
        private var traits: Traits = Traits(Address())

        //Constraints that outline when the work should be fired
        private val constraints = Constraints.Builder().setRequiredNetworkType(NetworkType.CONNECTED).build()

        //A worker object that handles sending offline events when the device is connected to the network
        private val fireStoredEventsWork = OneTimeWorkRequestBuilder<FireStoredEventsWork>()
            .setConstraints(constraints)
            .setInitialDelay(1, TimeUnit.HOURS)
            .build()

        private const val contentType = "application/json"

        //Retrofit service that will make the network calls, TODO : allow this to also take the BASE_URL
        private lateinit var event360Service: Event360Client
    }

//    private val appSigningKey by lazy {
//        getSigningKey(ctx)
//    }
//
//    private fun getSigningKey(context: Context): String {
//        try {
//            val info = context.packageManager.getPackageInfo(context.packageName, PackageManager.GET_SIGNATURES)
//            for (signature in info.signatures) {
//                val md = MessageDigest.getInstance("SHA-1")
//                md.update(signature.toByteArray())
//                return Base64.encodeToString(md.digest(), Base64.NO_WRAP)
//            }
//        } catch (nnf: PackageManager.NameNotFoundException) {
//            nnf.printStackTrace()
//        } catch (nsa: NoSuchAlgorithmException) {
//            nsa.printStackTrace()
//        }
//        return "No Signing key found!"
//    }

    //Returns the device resolution of the form width*height, TODO : improve this
    private val deviceScreenResolution by lazy {
        getDisplaySize(ctx)
    }

    /**
     * Returns the installer ID for the app
     */
    private val installerId by lazy {
        packageManager.getInstallerPackageName(applicationId) ?: "Unknown"
    }

    private val deviceLanguage by lazy {
        Locale.getDefault().displayLanguage
    }

    private val applicationName by lazy {
        packageManager
            .getApplicationLabel(
                packageManager
                    .getApplicationInfo(
                        applicationId,
                        PackageManager.GET_META_DATA
                    )
            ).toString()
    }

    private val deviceUserAgent by lazy {
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN_MR1) {
            WebSettings.getDefaultUserAgent(ctx)
        } else System.getProperty("http.agent")!!
    }

    /**
     * Returns if the device is a tablet of not
     * @return true if the device on which the app is running is a tablet, false otherwise
     */

    private fun isTablet(context: Context): Boolean {
        return context.resources.configuration.screenLayout and Configuration.SCREENLAYOUT_SIZE_MASK >= Configuration.SCREENLAYOUT_SIZE_LARGE
    }

    /**
     * Returns the display size of the phone on which the library is being used
     * @return Display size of the format width * height, TODO : improve the logic
     */

    private fun getDisplaySize(context: Context): String {
        val configuration = context.resources.configuration
        val screenWidthDp = configuration.screenWidthDp
        val screenHeightDp = configuration.screenHeightDp

        return "$screenWidthDp x $screenHeightDp"
    }

    /**
     * Channel of the device Tablet for a tabler, Mobile Phone for a mobile device
     */
    private val deviceChannel by lazy {
        if (isTablet(ctx)) "Tablet" else "Mobile Phone"
    }

    /**
     * Package name of the app
     */

    private val applicationId by lazy {
        ctx.packageName
    }

    private val packageManager by lazy {
        ctx.packageManager
    }

    private val pref: SharedPreferences by lazy {
        ctx.getSharedPreferences("event360", MODE_PRIVATE)
    }

    /**
     * Callback that's invoked every time a new activity's lifecycle method was called
     */
    private val activityLifecycleCallbacks = object : Application.ActivityLifecycleCallbacks {
        override fun onActivityPaused(activity: Activity?) {
        }

        override fun onActivityResumed(activity: Activity?) {
        }

        override fun onActivityStarted(activity: Activity?) {
        }

        override fun onActivityDestroyed(activity: Activity?) {
        }

        override fun onActivitySaveInstanceState(activity: Activity?, outState: Bundle?) {
        }

        /**
         * When an activity was stopped, fire a worker that retries the requests stored in the database
         */
        override fun onActivityStopped(activity: Activity?) {
            WorkManager
                .getInstance()
                .enqueueUniqueWork(
                    "fire-stored-events",
                    ExistingWorkPolicy.REPLACE,
                    fireStoredEventsWork
                )
        }

        /**
         * Called whenever an activity's onCreate method was called, automatically sends a pageview call to the server
         */
        override fun onActivityCreated(activity: Activity?, savedInstanceState: Bundle?) {
            activity?.let { ctx = it }
            track("${activity?.localClassName.toString()} Started")
        }
    }

    /**
     * This method should be called once in the Application class of the app which is using this library.
     * This method Initializes the library by :
     * 1. Storing the API_KEY to the companion object key
     * 2. Initializing the event360Service variable in companion object
     * 2. Registers the activityLifecycleCallbacks listener so that the library receives callbacks for each activity's lifecycle method
     * 3. Creates the database and saves it to the companion object eventDb
     * 4. Calls @link sendVerify with the API key to validate the API KEY
     *
     * @param apiKey : API KEY to be provided by the developer
     * @param url : BASE_URL for the appengine endpoint provided to the developer by solution360
     */

    fun initialize(apiKey: String, url: String) {
        event360Service = ClientGenerator.createService(Event360Client::class.java, url)

        key = apiKey
        (ctx.applicationContext as Application).registerActivityLifecycleCallbacks(activityLifecycleCallbacks)
        eventDb = Room.databaseBuilder(
            ctx,
            EventDatabase::class.java, "event360-db"
        ).build()

        sendVerify(apiKey)

    }

    private fun sendVerify(key: String) {
        event360Service.verify(contentType, Companion.key!!)
            .enqueue(object : Callback<ResponseBody> {
                override fun onFailure(call: Call<ResponseBody>, t: Throwable) = Unit

                override fun onResponse(call: Call<ResponseBody>, response: Response<ResponseBody>) {
                    if (response.code() == 403 || response.code() == 502) {
                        Log.e(TAG, "Invalid API key, please ensure that you are using the correct API key")
                        Companion.key = null
                    }
                }
            })
    }


    private fun getAnonymousID(): String {
        val uuid: String
        return if (pref.getString("anonymous_id", "").isEmpty()) {
            uuid = UUID.randomUUID().toString()
            pref.edit().putString("anonymous_id", uuid).apply()
            uuid
        } else {
            pref.getString("anonymous_id", "")
        }
    }

    private fun generateMeta(): Payload {
        val message = Payload()
        message.channel = deviceChannel
        with(message.context) {
            appId = applicationId
            language = deviceLanguage
            screenResolution = deviceScreenResolution
            operatingSystem = "Android"
            operatingSystemVersion = Build.VERSION.RELEASE
            mobileDeviceBranding = Build.BRAND
            mobileDeviceModel = Build.MODEL
            mobileDeviceMarketingName = Build.PRODUCT
            appInstallerId = installerId
            userAgent = deviceUserAgent
            appName = applicationName
            appVersion = BuildConfig.VERSION_NAME
            screenName = ctx.javaClass.simpleName
        }
        message.messageId = UUID.randomUUID().toString()
        message.pageInfo.title = ctx.javaClass.simpleName
        message.timestamp = TimeUnit.MILLISECONDS.toSeconds(System.currentTimeMillis()).toString()
        message.traits = traits
        message.version = BuildConfig.VERSION_NAME
        message.anonymousId = getAnonymousID()
        message.apiKey = key.toString()
        return message
    }

    fun track(eventName: String, extras: Extra) {
        val message = generateMeta()
        message.event.name = eventName
        message.extra = extras
        message.traits = traits
        sendEvent(message) {}
    }

    fun track(eventName: String) {
        val message = generateMeta()
        message.event.name = eventName
        message.traits = traits
        sendEvent(message) {}
    }

    internal fun sendEvent(message: Payload, result: (Boolean) -> Unit) {
        key?.let {
            if (isNetwork(ctx)) {
                message.sentAt = TimeUnit.MILLISECONDS.toSeconds(System.currentTimeMillis()).toString()
                event360Service.publish(contentType, message)
                    .enqueue(object : Callback<ResponseBody> {
                        override fun onFailure(call: Call<ResponseBody>, t: Throwable) {
                            t.printStackTrace()
                            saveEventInDb(message)
                        }

                        override fun onResponse(call: Call<ResponseBody>, response: Response<ResponseBody>) {
                            result(true)
                        }
                    })
            } else {
                saveEventInDb(message)
            }
        } ?: kotlin.run {
            Log.e(TAG, "Init Failed : Please ensure that you've initialized the library properly")
        }
    }

    private fun saveEventInDb(message: Payload) {
        thread {
            eventDb?.eventDao()?.insert(message)
        }
    }

    fun setTraits(name: String, email: String, userId: String, address: Address) {
        traits = Traits()
        with(traits) {
            this.name = name
            this.email = email
            this.userId = userId
            this.address = address
        }
    }
}

internal fun isNetwork(context: Context): Boolean {
    val connectivityManager = context.getSystemService(Context.CONNECTIVITY_SERVICE) as ConnectivityManager
    val activeNetworkInfo = connectivityManager.activeNetworkInfo
    return activeNetworkInfo != null && activeNetworkInfo.isConnected
}