package annguyen.kotlin.request

import annguyen.kotlin.request.json.JSONRequest
import annguyen.kotlin.request.promise.promise
import com.android.volley.Request
import com.android.volley.RequestQueue
import com.android.volley.toolbox.*
import org.jdeferred.Promise
import org.json.JSONArray
import org.json.JSONObject
import java.net.URLEncoder
import java.util.concurrent.ExecutionException
import java.util.concurrent.TimeUnit
import java.util.concurrent.TimeoutException

/**
 * Created by annguyen on 1/31/18.
 */
object Request {
    var mRequestQueue: RequestQueue

    init {
        mRequestQueue = init()
    }

    private fun init(): RequestQueue {
        mRequestQueue = RequestQueue(NoCache(), BasicNetwork(HurlStack()))
        mRequestQueue.start()
        return mRequestQueue
    }

    fun get(uri: String, params: JSONObject): Promise<JSONObject, Exception, Void> {
        val parsedURI = "$uri?${encodeParams(params)}"
        return sendRequest(uri = parsedURI, method = Request.Method.GET, params = JSONObject())
    }

    fun post(uri: String, params: JSONObject): Promise<JSONObject, Exception, Void> {
        return sendRequest(uri = uri, method = Request.Method.POST, params = params)
    }

    fun put(uri: String, params: JSONObject): Promise<JSONObject, Exception, Void> {
        return sendRequest(uri = uri, method = Request.Method.PUT, params = params)
    }

    fun delete(uri: String, params: JSONObject): Promise<JSONObject, Exception, Void> {
        return sendRequest(uri = uri, method = Request.Method.DELETE, params = params)
    }

    fun sendRequest(uri: String, method: Int, params: JSONObject): Promise<JSONObject, Exception, Void> {
        return promise {
            val future = RequestFuture.newFuture<JSONObject>()
            val request = JSONRequest(method, uri, params, future, future)
            mRequestQueue.add(request)
            try {
                val resp = future.get(10L, TimeUnit.SECONDS) as JSONObject
                resolve(resp)
            } catch (interruptedException: InterruptedException) {
                interruptedException.printStackTrace()
                reject(interruptedException)
            } catch (executionException: ExecutionException) {
                executionException.printStackTrace()
                reject(executionException)
            } catch (timeoutException: TimeoutException) {
                timeoutException.printStackTrace()
                reject(timeoutException)
            } catch (e: Exception) {
                e.printStackTrace()
                reject(e)
            }
        }
    }


    private fun encodeParams(params: JSONObject?): String {
        var paramString = ""
        if (params != null) {
            for (key in params.keys()) {
                val value = params[key]
                paramString += encodeParams(value, key, true)

            }
        }
        return paramString.dropLast(1)
    }

    private fun encodeParams(params: Any?, key: String, isRootKey: Boolean = true): String {
        var response = ""
        val parsedKey = URLEncoder.encode(if (isRootKey) key else "[$key]", "UTF-8")
        when (params) {
            is JSONObject -> for (paramKey in params.keys()) {
                response += "$parsedKey${encodeParams(params.get(paramKey), paramKey, false)}"
            }
            is JSONArray -> for (index in 0..(params.length() - 1)) {
                response += "$parsedKey${encodeParams(params.get(index), index.toString(), false)}"
            }
            else -> response += "$parsedKey=${URLEncoder.encode(params.toString(), "UTF-8")}&"
        }

        return response
    }

    private fun emptyResponse(exception: Exception): JSONObject {
        val jsonString = "{'status': 400, 'error': ${exception.message}}"
        return JSONObject(jsonString)
    }
}