package annguyen.music.equalizer;

import android.graphics.Color;
import android.media.MediaPlayer;
import android.media.audiofx.Equalizer;
import android.os.Bundle;
import android.support.annotation.Nullable;
import android.support.v4.app.Fragment;
import android.support.v4.content.ContextCompat;
import android.support.v7.widget.AppCompatSpinner;
import android.support.v7.widget.SwitchCompat;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.widget.CompoundButton;
import android.widget.LinearLayout;

import java.util.ArrayList;
import java.util.List;

/**
 * Created by annguyen on 10/27/17.
 */

public class EqualizerFragment extends Fragment implements EqualizerBar.OnFrequencyValueChangeListener, ChipsView.OnChipsSelectedChangeListener {

    private int mAudioSessionId;
    private Equalizer mEqualizer;

    private List<EqualizerBar> mListEqualizerBar;

    private AppCompatSpinner mViewListPreset;

    public static EqualizerFragment newInstance(MediaPlayer mMediaPlayer) {
        Bundle args = new Bundle();

        EqualizerFragment fragment = new EqualizerFragment();
        fragment.setAudioSessionId(mMediaPlayer.getAudioSessionId());
        fragment.setArguments(args);
        return fragment;
    }

    @Nullable
    @Override
    public View onCreateView(LayoutInflater inflater, @Nullable ViewGroup container, @Nullable Bundle savedInstanceState) {
        return inflater.inflate(R.layout.fragment_equalizer, container, false);
    }

    @Override
    public void onViewCreated(View view, @Nullable Bundle savedInstanceState) {
        super.onViewCreated(view, savedInstanceState);
        initialize();
    }

    /**
     * Set audio session id to equalizer
     *
     * @param audioSessionId {@link Integer} audioSessionId
     */
    public void setAudioSessionId(int audioSessionId) {
        mAudioSessionId = audioSessionId;
    }

    /**
     * Initialize fragment
     */
    private void initialize() {
        initialVariable();
        initialViewComponent();
    }

    /**
     * Initial variable of fragment
     */
    private void initialVariable() {
        // TODO init equalizer variable
        mEqualizer = new Equalizer(0, mAudioSessionId);

        mListEqualizerBar = new ArrayList<>();
    }

    /**
     * Init view component of fragment
     */
    private void initialViewComponent() {

        // TODO init equalizer component
        SwitchCompat mEqualizerSwitch = findViewById(R.id.view_switch_equalizer);
        final LinearLayout mEqualizerWrapperLayout = findViewById(R.id.view_equalizer_wrapper);
        mEqualizerSwitch.setOnCheckedChangeListener(new CompoundButton.OnCheckedChangeListener() {
            @Override
            public void onCheckedChanged(CompoundButton compoundButton, boolean isChecked) {
                // TODO enable or disable equalizer view
                mEqualizer.setEnabled(isChecked);
                enableDisableView(mEqualizerWrapperLayout, isChecked);
            }
        });


        short[] bandLevelRange = mEqualizer.getBandLevelRange();
        // TODO init expandable layout with equalizer
        for (short i = 0; i < mEqualizer.getNumberOfBands(); i++) {
            EqualizerBar equalizerBar = new EqualizerBar(getContext());
            equalizerBar.setFrequencyValueRange(bandLevelRange[0], bandLevelRange[1]);
            equalizerBar.setFrequencyNumber(mEqualizer.getCenterFreq(i) / 1000);
            equalizerBar.setOnFrequencyValueChangeListener(this);
            equalizerBar.setFrequencyBand(i);
            equalizerBar.setFrequencyNumber(mEqualizer.getBandLevel(i));

            mListEqualizerBar.add(equalizerBar);
            mEqualizerWrapperLayout.addView(equalizerBar);
        }

        // TODO init list preset equalizer
        ChipsView mViewListPresetWrapper = findViewById(R.id.view_list_preset_wrapper);
        for (short i = 0; i < mEqualizer.getNumberOfPresets(); i++) {
            CustomChip mChip = new CustomChip(getContext());
            mChip.setChipText(mEqualizer.getPresetName(i));
            mChip.changeSelectedBackgroundColor(ContextCompat.getColor(getContext(), R.color.colorPresetChipSelectedBackground));
            mChip.setSelectedTextColor(ContextCompat.getColor(getContext(), R.color.colorPresetChipSelectedText));
            mChip.changeBackgroundColor(ContextCompat.getColor(getContext(), R.color.colorPresetChipDefaultBackground));
            mChip.setTextColor(ContextCompat.getColor(getContext(), R.color.colorPresetChipDefaultText));
            mChip.setClosable(false);
            mChip.setPresetNumber(i);
            mViewListPresetWrapper.addChip(mChip.build());
        }
        mViewListPresetWrapper.setChipSelectedChangeListener(this);

        // TODO init data for first time init of equalizer
        mEqualizerSwitch.setChecked(mEqualizer.getEnabled());
        mViewListPresetWrapper.setSelectedChip(mEqualizer.getCurrentPreset());
        enableDisableView(mEqualizerWrapperLayout, mEqualizer.getEnabled());
    }

    /**
     * Support find view by view resource id for Fragment class
     *
     * @param viewResId {@link Integer} resource id of view
     * @param <T>       type of view
     * @return {@link View} View with type if root view is not null and root view has view with {@link Integer} viewResId, else return null
     */
    private <T extends View> T findViewById(int viewResId) {
        if (getView() != null) {
            return getView().findViewById(viewResId);
        }
        return null;
    }

    /**
     * Handle frequency number change
     *
     * @param band      {@link Short} band number
     * @param frequency {@link Short} frequency value
     */
    @Override
    public void onFrequencyValueChanged(short band, short frequency) {
        mEqualizer.setBandLevel(band, frequency);
    }

    /**
     * Update EqualizerBar when preset change
     */
    private void updateSeekBarOnPresetChange() {
        for (int i = 0; i < mListEqualizerBar.size(); i++) {
            EqualizerBar equalizerBar = mListEqualizerBar.get(i);
            equalizerBar.setFrequencyNumber(mEqualizer.getBandLevel(equalizerBar.getFrequencyBand()));
        }
    }

    /**
     * Enable or disable view and all child view
     *
     * @param view    {@link View} parent view
     * @param enabled If true, enable. Else disable
     */
    private void enableDisableView(View view, boolean enabled) {
        view.setEnabled(enabled);

        if (view instanceof ViewGroup) {
            ViewGroup group = (ViewGroup) view;

            for (int idx = 0; idx < group.getChildCount(); idx++) {
                enableDisableView(group.getChildAt(idx), enabled);
            }
        }
    }

    @Override
    public void onChipChange(CustomChip chip) {
        mEqualizer.usePreset(chip.getPresetNumber());
        updateSeekBarOnPresetChange();
    }
}
