package com.tool

import android.annotation.SuppressLint
import android.content.Context
import android.content.pm.ApplicationInfo
import android.content.pm.PackageManager
import android.graphics.drawable.Drawable
import android.util.TypedValue
import android.view.View
import android.view.ViewGroup
import android.widget.TextView

/**
 * android 常见的工具类
 * Created by hm on 2017/9/19.
 */
private const val KEY_DESIGN_WIDTH = "design_width"
private const val KEY_DESIGN_HEIGHT = "design_height"
private var scale: Float = 1f

/**
 * 初始化设置缩放因子
 */
fun Context.initScaleValue() {
    val displayMetrics = resources.displayMetrics
    val widthPixels = displayMetrics.widthPixels
    val heightPixels = displayMetrics.heightPixels
    val designWidthAndHeight = getDesignWidthAndHeight(this)
    val scaleX = widthPixels.toFloat() / designWidthAndHeight[0]
    val scaleY = heightPixels.toFloat() / designWidthAndHeight[1]
    scale = if (scaleX >= scaleY) scaleX else scaleY
}

/**
 * 获取设置的设计稿的宽高  如果没有设置则去默认值
 *
 * @param context 上下文
 * @return 宽高的数组
 */
@SuppressLint("WrongConstant")
private fun getDesignWidthAndHeight(context: Context): IntArray {
    val packageManager = context.packageManager
    val applicationInfo: ApplicationInfo
    val designWidthAndHeight = IntArray(2)
    designWidthAndHeight[0] = 1080
    designWidthAndHeight[1] = 1920
    applicationInfo = packageManager.getApplicationInfo(context.packageName, PackageManager.GET_META_DATA)
    applicationInfo.metaData?.let {
        designWidthAndHeight[0] = applicationInfo.metaData.get(KEY_DESIGN_WIDTH) as Int
        designWidthAndHeight[1] = applicationInfo.metaData.get(KEY_DESIGN_HEIGHT) as Int
    }
    return designWidthAndHeight
}

/**
 * view的缩放
 */
fun View.scale() = //如果是viewGroup则需要进行遍历子View进行缩放
        when (this) {
            is ViewGroup -> scaleViewGroup(this)
            else -> scaleView(this)
        }

/**
 * viewGroup的缩放
 */
private fun scaleViewGroup(viewGroup: ViewGroup) {
    for (i in 0 until viewGroup.childCount) {
        viewGroup.getChildAt(i).scale()
    }
}


/**
 * 缩放其他View
 */
private fun scaleView(view: View) {
    scaleViewSize(view)
    if (view is TextView) {//如果是TextView 还要缩放字体大小和设置的图片
        val size = view.textSize * scale
        view.setTextSize(TypedValue.COMPLEX_UNIT_PX, size)
        val drawables = view.compoundDrawables
        val leftDrawable = drawables[0]
        val topDrawable = drawables[1]
        val rightDrawable = drawables[2]
        val bottomDrawable = drawables[3]
        setTextViewCompoundDrawables(view, leftDrawable, topDrawable, rightDrawable, bottomDrawable)
        val compoundDrawablePadding = getScaleValue(view.compoundDrawablePadding)
        view.compoundDrawablePadding = compoundDrawablePadding
    }
}

private fun setTextViewCompoundDrawables(textView: TextView, leftDrawable: Drawable?
                                         , topDrawable: Drawable?, rightDrawable: Drawable?, bottomDrawable: Drawable?) {
    if (null != leftDrawable) {
        scaleDrawableBounds(leftDrawable)
    }
    if (null != rightDrawable) {
        scaleDrawableBounds(rightDrawable)
    }
    if (null != topDrawable) {
        scaleDrawableBounds(topDrawable)
    }
    if (null != bottomDrawable) {
        scaleDrawableBounds(bottomDrawable)
    }
    textView.setCompoundDrawables(leftDrawable, topDrawable, rightDrawable, bottomDrawable)
}

private fun scaleViewSize(view: View) {
    //缩放所有的内边距
    view.setPadding(getScaleValue(view.paddingLeft), getScaleValue(view.paddingTop)
            , getScaleValue(view.paddingRight), getScaleValue(view.paddingBottom))
    //缩放layoutParams
    val layoutParams = view.layoutParams
    layoutParams?.let {
        if (it.width > 0) {
            it.width = getScaleValue(it.width)
        }
        if (it.height > 0) {
            it.height = getScaleValue(it.height)
        }
        if (it is ViewGroup.MarginLayoutParams) {
            it.topMargin = getScaleValue(it.topMargin)
            it.leftMargin = getScaleValue(it.leftMargin)
            it.rightMargin = getScaleValue(it.rightMargin)
            it.bottomMargin = getScaleValue(it.bottomMargin)
        }
        view.layoutParams = it
    }
}

/**
 * 根据缩放因子计算缩放值
 */
private fun getScaleValue(value: Int): Int {
    return if (value <= 4) value else Math.ceil((scale * value.toFloat()).toDouble()).toInt()
}

private fun scaleDrawableBounds(drawable: Drawable) {
    val right = getScaleValue(drawable.intrinsicWidth)
    val bottom = getScaleValue(drawable.intrinsicHeight)
    drawable.setBounds(0, 0, right, bottom)
}


