package android.rapid.net;

import android.rapid.net.exception.ParseError;

import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.util.Collections;
import java.util.Map;

public abstract class Request<T> {
    private static final String DEFAULT_PARAMS_ENCODING = "UTF-8";

    private final int mMethod;

    private final String mUrl;

    public Request(String url) {
        this(Method.GET, url);
    }

    public Request(int method, String url) {
        mMethod = method;
        mUrl = url;
    }

    abstract protected Response<T> parseNetworkResponse(NetworkResponse response) throws ParseError;

    public int getMethod() {
        return mMethod;
    }

    public String getUrl() {
        return mUrl;
    }

    protected String getParamsEncoding() {
        return DEFAULT_PARAMS_ENCODING;
    }

    public String getBodyContentType() {
        return "application/x-www-form-urlencoded; charset=" + getParamsEncoding();
    }

    protected Map<String, String> getParams() {
        return null;
    }

    public byte[] getBody() {
        Map<String, String> params = getParams();
        if (params != null && params.size() > 0) {
            return encodeParameters(params, getParamsEncoding());
        }
        return null;
    }

    public int getTimeoutMs() {
        return 5000;
    }

    public Map<String, String> getHeaders() {
        return Collections.emptyMap();
    }

    private byte[] encodeParameters(Map<String, String> params, String paramsEncoding) {
        StringBuilder encodedParams = new StringBuilder();
        try {
            for (Map.Entry<String, String> entry : params.entrySet()) {
                encodedParams.append(URLEncoder.encode(entry.getKey(), paramsEncoding));
                encodedParams.append('=');
                encodedParams.append(URLEncoder.encode(entry.getValue(), paramsEncoding));
                encodedParams.append('&');
            }
            return encodedParams.toString().getBytes(paramsEncoding);
        } catch (UnsupportedEncodingException uee) {
            throw new RuntimeException("Encoding not supported: " + paramsEncoding, uee);
        }
    }

    public interface Method {
        int GET = 0;
        int POST = 1;
    }
}
