package android.rapid.connect.wifi;

import android.content.Context;
import android.content.IntentFilter;
import android.net.wifi.WpsInfo;
import android.net.wifi.p2p.WifiP2pConfig;
import android.net.wifi.p2p.WifiP2pDevice;
import android.net.wifi.p2p.WifiP2pDeviceList;
import android.net.wifi.p2p.WifiP2pInfo;
import android.net.wifi.p2p.WifiP2pManager;
import android.net.wifi.p2p.WifiP2pManager.Channel;
import android.net.wifi.p2p.WifiP2pManager.ConnectionInfoListener;
import android.net.wifi.p2p.WifiP2pManager.PeerListListener;
import android.os.Handler;
import android.rapid.connect.ConnectDevice;
import android.rapid.connect.IAvailableDeviceListener;
import android.rapid.connect.AbsConnect;

import java.util.ArrayList;
import java.util.List;

public class raWifiConnect extends AbsConnect<WifiP2pDevice> implements ConnectionInfoListener, PeerListListener {
    private Context mContext;
    private WifiP2pManager mWifiP2pManager;
    private Channel mChannel;
    private WiFiDirectBroadcastReceiver mReceiver;
    private List<ConnectDevice> mDevices = new ArrayList<ConnectDevice>();
    private IAvailableDeviceListener mAvailableDeviceListener;
    private WifiSocketService mSocketService;
    private ConnectDevice mThisDevice;

    public raWifiConnect(Context context, Handler handler) {
        mContext = context;
        mSocketService = new WifiSocketService(handler);
        mWifiP2pManager = (WifiP2pManager) context.getSystemService(Context.WIFI_P2P_SERVICE);
        mChannel = mWifiP2pManager.initialize(context, context.getMainLooper(), null);
    }

    @Override
    public void enable() {
        IntentFilter intentFilter = new IntentFilter();
        intentFilter.addAction(WifiP2pManager.WIFI_P2P_STATE_CHANGED_ACTION);
        intentFilter.addAction(WifiP2pManager.WIFI_P2P_PEERS_CHANGED_ACTION);
        intentFilter.addAction(WifiP2pManager.WIFI_P2P_CONNECTION_CHANGED_ACTION);
        intentFilter.addAction(WifiP2pManager.WIFI_P2P_THIS_DEVICE_CHANGED_ACTION);
        mReceiver = new WiFiDirectBroadcastReceiver(this);
        mContext.registerReceiver(mReceiver, intentFilter);
    }

    @Override
    public void disable() {
        mContext.unregisterReceiver(mReceiver);
        mSocketService.stop();
    }

    @Override
    public void searchDevcie(IAvailableDeviceListener listener) {
        mAvailableDeviceListener = listener;
        discoverPeers();
    }

    @Override
    public void connect(final ConnectDevice<WifiP2pDevice> device) {
        WifiP2pConfig config = new WifiP2pConfig();
        config.deviceAddress = device.device.deviceAddress;
        config.wps.setup = WpsInfo.PBC;
        mWifiP2pManager.connect(mChannel, config, null);
    }

    @Override
    public void cancelConnect() {
        if (isConnected()) {
            mWifiP2pManager.removeGroup(mChannel, null);
        } else {
            mWifiP2pManager.cancelConnect(mChannel, null);
        }
        mSocketService.stop();
    }

    @Override
    public boolean isConnected() {
        return mSocketService.getState() == WifiSocketService.STATE_CONNECTED;
    }

    @Override
    public void send(byte[] data) {
        if (isConnected()) {
            mSocketService.write(data);
        }
    }

    @Override
    public List<ConnectDevice> getAvailableDevices() {
        return mDevices;
    }

    public void discoverPeers() {
        mWifiP2pManager.discoverPeers(mChannel, null);
    }

    public void resetPeers() {
        mDevices.clear();
    }

    public boolean isWifiP2pAviliable() {
        return mWifiP2pManager != null;
    }

    public void requestPeers() {
        mWifiP2pManager.requestPeers(mChannel, this);
    }

    public void requestConnectionInfo() {
        mWifiP2pManager.requestConnectionInfo(mChannel, this);
    }

    @Override
    public void onPeersAvailable(WifiP2pDeviceList peers) {
        resetPeers();
        for (WifiP2pDevice wifidevice : peers.getDeviceList()) {
            ConnectDevice<WifiP2pDevice> device = new ConnectDevice<>();
            device.deviceName = wifidevice.deviceName;
            device.device = wifidevice;
            device.status = wifidevice.status;
            device.connected = (wifidevice.status == WifiP2pDevice.CONNECTED);
            device.deviceStatus = getDeviceStatus(wifidevice.status);
            mDevices.add(device);
        }
        if (mAvailableDeviceListener != null) {
            mAvailableDeviceListener.availableDevice(mDevices);
        }
    }

    @Override
    public void onConnectionInfoAvailable(WifiP2pInfo info) {
        if (info.groupFormed && info.isGroupOwner) {
            if (mSocketService.getState() == WifiSocketService.STATE_NONE) {
                mSocketService.start();
            }
        } else if (info.groupFormed) {
            mSocketService.connect(info.groupOwnerAddress.getHostAddress());
        }
    }

    private String getDeviceStatus(int deviceStatus) {
        switch (deviceStatus) {
            case WifiP2pDevice.AVAILABLE:
                return "Available";
            case WifiP2pDevice.INVITED:
                return "Invited";
            case WifiP2pDevice.CONNECTED:
                return "Connected";
            case WifiP2pDevice.FAILED:
                return "Failed";
            case WifiP2pDevice.UNAVAILABLE:
                return "Unavailable";
            default:
                return "Unknown";
        }
    }

    public void updateThisDevice(WifiP2pDevice wifidevice) {
        mThisDevice = new ConnectDevice<WifiP2pDevice>();
        mThisDevice.device = wifidevice;
        mThisDevice.deviceName = wifidevice.deviceName;
        mThisDevice.status = wifidevice.status;
        mThisDevice.deviceStatus = getDeviceStatus(wifidevice.status);
    }

    public ConnectDevice<WifiP2pDevice> getDevice() {
        return mThisDevice;
    }
}
