package android.rapid.connect.bluetooth;

import android.bluetooth.BluetoothAdapter;
import android.bluetooth.BluetoothDevice;
import android.bluetooth.BluetoothManager;
import android.content.Context;
import android.content.IntentFilter;
import android.os.Handler;
import android.rapid.connect.ConnectDevice;
import android.rapid.connect.IAvailableDeviceListener;
import android.rapid.connect.AbsConnect;

import java.util.ArrayList;
import java.util.List;

public class raBluetoothConnect extends AbsConnect<BluetoothDevice> {
    private final BluetoothAdapter mAdapter;
    private final BluetoothManager mBluetoothManager;
    private Context mContext;
    private BluetoothSocketService mSocketService;
    private IAvailableDeviceListener mAvailableDeviceListener;
    private BluetoothBroadcastReceiver mReceiver;
    private List<ConnectDevice> mConnectDevices = new ArrayList<ConnectDevice>();
    private List<ConnectDevice> mNewDevices = new ArrayList<ConnectDevice>();

    public raBluetoothConnect(Context context, Handler handler) {
        mContext = context;
        mBluetoothManager = (BluetoothManager) context.getSystemService(Context.BLUETOOTH_SERVICE);
        mAdapter = mBluetoothManager.getAdapter();
        mSocketService = new BluetoothSocketService(mAdapter, handler);
        mReceiver = new BluetoothBroadcastReceiver(this);
        for (BluetoothDevice btdevice : mAdapter.getBondedDevices()) {
            ConnectDevice device = new ConnectDevice<BluetoothDevice>();
            device.device = btdevice;
            device.deviceName = btdevice.getName();
            device.status = btdevice.getBondState();
            device.connected = (btdevice.getBondState() == BluetoothDevice.BOND_BONDED);
            device.deviceStatus = getDeviceStatus(btdevice.getBondState());
            mConnectDevices.add(device);
        }
    }

    @Override
    public void enable() {
        IntentFilter intentFilter = new IntentFilter();
        intentFilter.addAction(BluetoothDevice.ACTION_FOUND);
        mContext.registerReceiver(mReceiver, intentFilter);
        if (mAdapter.isEnabled()) {
            if (mSocketService.getState() == BluetoothSocketService.STATE_NONE) {
                mSocketService.start();
            }
        } else {
            mAdapter.enable();
        }
    }

    @Override
    public void disable() {
        if (mAdapter != null) {
            mAdapter.cancelDiscovery();
        }
        mContext.unregisterReceiver(mReceiver);
        mSocketService.stop();
    }

    @Override
    public void searchDevcie(IAvailableDeviceListener listener) {
        if (mAdapter.isDiscovering()) {
            mAdapter.startDiscovery();
        }
        mAdapter.startDiscovery();
        mAvailableDeviceListener = listener;
    }

    @Override
    public void connect(ConnectDevice<BluetoothDevice> device) {
        mSocketService.connect(device.device);
    }

    @Override
    public void cancelConnect() {
        mSocketService.stop();
    }

    @Override
    public boolean isConnected() {
        return mSocketService.getState() == BluetoothSocketService.STATE_CONNECTED;
    }

    @Override
    public void send(byte[] data) {
        if (isConnected()) {
            mSocketService.write(data);
        }
    }

    @Override
    public List<ConnectDevice> getAvailableDevices() {
        return mConnectDevices;
    }

    private String getDeviceStatus(int deviceStatus) {
        switch (deviceStatus) {
            case BluetoothDevice.BOND_NONE:
                return "NONE";
            case BluetoothDevice.BOND_BONDING:
                return "BONDING";
            case BluetoothDevice.BOND_BONDED:
                return "BONDED";
            default:
                return "Unknown";
        }
    }

    public void addNewDevice(BluetoothDevice device) {
        ConnectDevice btdevice = new ConnectDevice<BluetoothDevice>();
        btdevice.device = device;
        btdevice.deviceName = device.getName();
        btdevice.status = device.getBondState();
        btdevice.deviceStatus = getDeviceStatus(device.getBondState());
        if (device.getBondState() != BluetoothDevice.BOND_BONDED) {
            mNewDevices.add(btdevice);
        }
        if (mAvailableDeviceListener != null) {
            mAvailableDeviceListener.nextDevice(btdevice);
        }
    }
}
