package android.rapid.task;

import android.os.Handler;
import android.os.Looper;
import android.os.Message;
import android.rapid.log.LogUtil;

import java.util.ArrayDeque;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.Callable;
import java.util.concurrent.CancellationException;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Executor;
import java.util.concurrent.FutureTask;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;

public final class raAsyncTask<Params, Result> {
    private static final LogUtil.Tag TAG = new LogUtil.Tag("raAsyncTask");
    private raResult<Result> mTarget;

    private WorkerRunnable<Params, Result> mWorker;
    private FutureTask<Result> mFuture;
    public static final Executor SERIAL_EXECUTOR = new SerialExecutor();
    private static volatile Executor sDefaultExecutor = SERIAL_EXECUTOR;
    private static final int CPU_COUNT = Runtime.getRuntime()
            .availableProcessors();
    private static final int CORE_POOL_SIZE = CPU_COUNT + 1;
    private static final int MAXIMUM_POOL_SIZE = CPU_COUNT * 2 + 1;
    private static final int KEEP_ALIVE = 1;

    private static final int MESSAGE_POST_RESULT = 0x1;

    private static final ThreadFactory sThreadFactory = new ThreadFactory() {
        private final AtomicInteger mCount = new AtomicInteger(1);

        public Thread newThread(Runnable r) {
            return new Thread(r, "raAsyncTask #" + mCount.getAndIncrement());
        }
    };

    private static final BlockingQueue<Runnable> sPoolWorkQueue = new LinkedBlockingQueue<Runnable>(
            128);

    public static final Executor THREAD_POOL_EXECUTOR = new ThreadPoolExecutor(
            CORE_POOL_SIZE, MAXIMUM_POOL_SIZE, KEEP_ALIVE, TimeUnit.SECONDS,
            sPoolWorkQueue, sThreadFactory);

    private static InternalHandler sHandler;

    private final AtomicBoolean mCancelled = new AtomicBoolean();
    private final AtomicBoolean mTaskInvoked = new AtomicBoolean();

    private raAsyncTask(final raTask<Params, Result> task, final Params params) {
        this.mWorker = new WorkerRunnable<Params, Result>() {
            public Result call() throws Exception {
                return postResult(task.execute(params));
            }
        };
        this.mWorker.mParams = params;
    }

    /**
     *
     * @param task raTask object
     * @param params params
     * @param <Params> Params class type
     * @param <Result> Result class type
     * @return raAsyncTask object
     */
    public static <Params, Result> raAsyncTask<Params, Result> createTask(
            raTask<Params, Result> task, Params params) {
        return new raAsyncTask<Params, Result>(task, params);
    }

    /**
     * Connect raTask object to raTarget
     * @param target raAsyncTask.raTarget object
     */
    public void subscibe(final raResult<Result> target) {
        mFuture = new FutureTask<Result>(mWorker) {
            @Override
            protected void done() {
                try {
                    postResultIfNotInvoked(get());
                } catch (InterruptedException e) {
                    LogUtil.e(TAG, "InterruptedException", e);
                } catch (ExecutionException e) {
                    target.onError(e);
                } catch (CancellationException e) {
                    postResultIfNotInvoked(null);
                }
            }
        };
        this.mTarget = target;
        sDefaultExecutor.execute(mFuture);
    }

    private final boolean isCancelled() {
        return mCancelled.get();
    }

    public final boolean cancel() {
        mCancelled.set(true);
        return mFuture.cancel(true);
    }

    private static Handler getHandler() {
        synchronized (raAsyncTask.class) {
            if (sHandler == null) {
                sHandler = new InternalHandler();
            }
            return sHandler;
        }
    }

    private void postResultIfNotInvoked(Result result) {
        final boolean wasTaskInvoked = mTaskInvoked.get();
        if (!wasTaskInvoked) {
            postResult(result);
        }
    }

    private Result postResult(Result result) {
        Message message = getHandler().obtainMessage(MESSAGE_POST_RESULT,
                new AsyncTaskResult<Result>(this, result));
        message.sendToTarget();
        return result;
    }

    private void finish(Result result) {
        if (isCancelled()) {
            mTarget.onCancelled(result);
        } else {
            mTarget.onResult(result);
        }
    }

    public interface raTask<Params, Result> {
        Result execute(Params params) throws Exception;
    }

    public interface raResult<Result> {
        void onCancelled(Result result);

        void onError(Throwable e);

        void onResult(Result result);
    }


    private static abstract class WorkerRunnable<Params, Result> implements
            Callable<Result> {
        Params mParams;
    }

    private static class InternalHandler extends Handler {
        public InternalHandler() {
            super(Looper.getMainLooper());
        }

        @Override
        public void handleMessage(Message msg) {
            AsyncTaskResult<?> result = (AsyncTaskResult<?>) msg.obj;
            switch (msg.what) {
                case MESSAGE_POST_RESULT:
                    result.mTask.finish(result.mData);
                    break;
            }
        }
    }

    private static class AsyncTaskResult<Data> {
        final raAsyncTask mTask;
        final Data mData;

        AsyncTaskResult(raAsyncTask task, Data data) {
            mTask = task;
            mData = data;
        }
    }

    private static class SerialExecutor implements Executor {
        final ArrayDeque<Runnable> mTasks = new ArrayDeque<Runnable>();
        Runnable mActive;

        public synchronized void execute(final Runnable r) {
            mTasks.offer(new Runnable() {
                public void run() {
                    try {
                        r.run();
                    } finally {
                        scheduleNext();
                    }
                }
            });
            if (mActive == null) {
                scheduleNext();
            }
        }

        protected synchronized void scheduleNext() {
            if ((mActive = mTasks.poll()) != null) {
                THREAD_POOL_EXECUTOR.execute(mActive);
            }
        }
    }
}