package android.rapid.connect.socket;

import android.rapid.connect.IDispatchListener;
import android.rapid.log.LogUtil;

import java.io.IOException;
import java.net.InetSocketAddress;
import java.net.Socket;

public class raSocketService {
    public static final int STATE_NONE = 0;       // we're doing nothing
    public static final int STATE_LISTEN = 1;     // now listening for incoming connections
    public static final int STATE_CONNECTING = 2; // now initiating an outgoing connection
    public static final int STATE_CONNECTED = 3;  // now connected to a remote device
    private static final LogUtil.Tag TAG = new LogUtil.Tag("SocketService");
    private final IDispatchListener mDispatchListener;
    protected AbsThread mAcceptThread;
    protected AbsThread mConnectThread;
    protected AbsThread mConnectedThread;
    private int mState;

    public raSocketService(IDispatchListener handler) {
        mState = STATE_NONE;
        mDispatchListener = handler;
    }

    public synchronized int getState() {
        return mState;
    }

    protected synchronized void setState(int state) {
        LogUtil.d(TAG, "setState() " + mState + " -> " + state);
        mState = state;
    }

    public synchronized void start() {
        LogUtil.d(TAG, "start");

        if (mConnectThread != null) {
            mConnectThread.cancel();
            mConnectThread = null;
        }

        if (mConnectedThread != null) {
            mConnectedThread.cancel();
            mConnectedThread = null;
        }

        setState(STATE_LISTEN);

        if (mAcceptThread == null) {
            mAcceptThread = new AcceptThread();
            mAcceptThread.start();
        }
    }

    public synchronized void connect(String address) {
        LogUtil.d(TAG, "connect to: " + address);

        if (mState == STATE_CONNECTING) {
            if (mConnectThread != null) {
                mConnectThread.cancel();
                mConnectThread = null;
            }
        }

        if (mConnectedThread != null) {
            mConnectedThread.cancel();
            mConnectedThread = null;
        }

        mConnectThread = new ConnectThread(address);
        mConnectThread.start();
        setState(STATE_CONNECTING);
    }

    public synchronized void connected(Socket socket) {
        cancelThread();

        mConnectedThread = new ConnectedThread(socket);
        mConnectedThread.start();

        setState(STATE_CONNECTED);
    }

    public synchronized void stop() {
        LogUtil.d(TAG, "stop");
        cancelThread();
        setState(STATE_NONE);
    }

    private void cancelThread() {
        if (mConnectThread != null) {
            mConnectThread.cancel();
            mConnectThread = null;
        }

        if (mConnectedThread != null) {
            mConnectedThread.cancel();
            mConnectedThread = null;
        }

        if (mAcceptThread != null) {
            mAcceptThread.cancel();
        }
    }

    public void write(byte[] out) {
        DispatchThread r;
        synchronized (this) {
            if (mState != STATE_CONNECTED) return;
            r = (DispatchThread) mConnectedThread;
        }
        r.write(out);
    }

    protected void connectionFailed() {
        mDispatchListener.connectionFailed();
        start();
    }

    protected void connectionLost() {
        mDispatchListener.connectionLost();
        start();
    }

    protected void dispatchRecv(int length, byte[] buff) {
        mDispatchListener.recv(length, buff);
    }

    private class AcceptThread extends ServerThread {
        @Override
        public void execute() {
            Socket socket = null;

            while (mState != STATE_CONNECTED) {
                try {
                    socket = mServerSocket.accept();
                } catch (IOException e) {
                    LogUtil.e(TAG, "Exception during accept", e);
                    break;
                }

                if (socket != null) {
                    synchronized (this) {
                        switch (mState) {
                            case STATE_LISTEN:
                            case STATE_CONNECTING:
                                connected(socket);
                                break;
                            case STATE_NONE:
                            case STATE_CONNECTED:
                                try {
                                    socket.close();
                                } catch (IOException e) {
                                    LogUtil.e(TAG, "Exception during close", e);
                                }
                                break;
                        }
                    }
                }
                LogUtil.i(TAG, "end run()");
            }
        }
    }

    private class ConnectThread extends ClientThread {
        public ConnectThread(String address) {
            super(address);
        }

        @Override
        public void execute() {
            LogUtil.i(TAG, "start run()");

            try {
                mSocket.connect((new InetSocketAddress(mAddress, mServerPort)), 0);
            } catch (IOException e) {
                try {
                    mSocket.close();
                } catch (IOException e2) {
                    LogUtil.e(TAG, "Exception during close", e2);
                }
                connectionFailed();
                return;
            }

            synchronized (this) {
                mConnectThread = null;
            }

            connected(mSocket);
        }
    }

    private class ConnectedThread extends DispatchThread {
        public ConnectedThread(Socket socket) {
            super(socket);
        }

        @Override
        public void execute() {
            LogUtil.i(TAG, "start run()");
            byte[] buffer = new byte[1024];
            int bytes;

            while (true) {
                try {
                    bytes = mInStream.read(buffer);
                    dispatchRecv(bytes, buffer);
                } catch (IOException e) {
                    LogUtil.e(TAG, "disconnected");
                    connectionLost();
                    raSocketService.this.start();
                    break;
                }
            }
        }
    }
}
